/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.service;

import com.datmt.pdftools.model.JoinerSection;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javafx.embed.swing.SwingFXUtils;
import javafx.scene.image.Image;
import javafx.scene.image.WritableImage;
import javax.imageio.ImageIO;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.io.RandomAccessReadBufferedFile;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.image.LosslessFactory;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfBulkInserterService {
    private static final Logger logger = LoggerFactory.getLogger(PdfBulkInserterService.class);
    private static final int THUMBNAIL_DPI = 72;

    public PDDocument loadSourcePdf(File file) throws IOException {
        logger.info("Loading source PDF: {}", (Object)file.getName());
        return Loader.loadPDF(new RandomAccessReadBufferedFile(file));
    }

    public PDDocument createPdfFromImage(File imageFile, InsertionOptions options) throws IOException {
        logger.info("Creating PDF from image: {}", (Object)imageFile.getName());
        BufferedImage bufferedImage = ImageIO.read(imageFile);
        if (bufferedImage == null) {
            throw new IOException("Failed to read image: " + imageFile.getName());
        }
        PDDocument document = new PDDocument();
        PDRectangle pageSize = this.getPageSize(options.getPageSize(), bufferedImage);
        PDPage page = new PDPage(pageSize);
        document.addPage(page);
        float[] placement = this.calculateImagePlacement(bufferedImage, pageSize, options.getFitOption());
        float x = placement[0];
        float y = placement[1];
        float width = placement[2];
        float height = placement[3];
        PDImageXObject pdImage = LosslessFactory.createFromImage(document, bufferedImage);
        try (PDPageContentStream contentStream = new PDPageContentStream(document, page);){
            contentStream.drawImage(pdImage, x, y, width, height);
        }
        return document;
    }

    public int getPageCount(File pdfFile) throws IOException {
        try (PDDocument doc = Loader.loadPDF(new RandomAccessReadBufferedFile(pdfFile));){
            int n = doc.getNumberOfPages();
            return n;
        }
    }

    public Image renderThumbnail(File file, boolean isPdf, int pageIndex) throws IOException {
        if (isPdf) {
            try (PDDocument doc = Loader.loadPDF(new RandomAccessReadBufferedFile(file));){
                PDFRenderer renderer = new PDFRenderer(doc);
                BufferedImage bufferedImage = renderer.renderImageWithDPI(pageIndex, 72.0f, ImageType.RGB);
                WritableImage writableImage = SwingFXUtils.toFXImage(bufferedImage, null);
                return writableImage;
            }
        }
        return new Image(file.toURI().toString());
    }

    public Image renderThumbnailFromDocument(PDDocument doc, int pageIndex) throws IOException {
        PDFRenderer renderer = new PDFRenderer(doc);
        BufferedImage bufferedImage = renderer.renderImageWithDPI(pageIndex, 72.0f, ImageType.RGB);
        return SwingFXUtils.toFXImage(bufferedImage, null);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public InsertResult insertIntoTarget(File targetFile, PDDocument sourceDoc, InsertionOptions options) {
        InsertResult result = new InsertResult(targetFile);
        try (PDDocument targetDoc = Loader.loadPDF(new RandomAccessReadBufferedFile(targetFile));){
            int originalPageCount = targetDoc.getNumberOfPages();
            result.setOriginalPageCount(originalPageCount);
            List<Integer> positions = this.calculateInsertionPositions(originalPageCount, options);
            if (positions.isEmpty()) {
                result.setSuccess(true);
                result.setInsertionsPerformed(0);
                result.setNewPageCount(originalPageCount);
                result.setOutputFile(targetFile);
                InsertResult insertResult = result;
                return insertResult;
            }
            try (PDDocument newDoc = new PDDocument();){
                int sourcePageCount = sourceDoc.getNumberOfPages();
                int insertionsPerformed = 0;
                int positionIndex = 0;
                int currentPosition = positions.get(positionIndex);
                for (int i = 0; i < originalPageCount; ++i) {
                    PDPage targetPage = targetDoc.getPage(i);
                    newDoc.importPage(targetPage);
                    if (positionIndex >= positions.size() || i + 1 != currentPosition) continue;
                    for (int j = 0; j < sourcePageCount; ++j) {
                        PDPage sourcePage = sourceDoc.getPage(j);
                        newDoc.importPage(sourcePage);
                    }
                    ++insertionsPerformed;
                    if (++positionIndex >= positions.size()) continue;
                    currentPosition = positions.get(positionIndex);
                }
                while (positionIndex < positions.size()) {
                    for (int j = 0; j < sourcePageCount; ++j) {
                        PDPage sourcePage = sourceDoc.getPage(j);
                        newDoc.importPage(sourcePage);
                    }
                    ++insertionsPerformed;
                    ++positionIndex;
                }
                File outputFile = this.generateOutputFile(targetFile, options.getOutputSuffix());
                result.setOutputFile(outputFile);
                newDoc.save(outputFile);
                result.setSuccess(true);
                result.setInsertionsPerformed(insertionsPerformed);
                result.setNewPageCount(newDoc.getNumberOfPages());
                logger.info("Inserted {} time(s) into {}: {} -> {} pages", insertionsPerformed, targetFile.getName(), originalPageCount, result.getNewPageCount());
                return result;
            }
        }
        catch (IOException e) {
            logger.error("Error inserting into {}: {}", (Object)targetFile.getName(), (Object)e.getMessage());
            result.setSuccess(false);
            result.setErrorMessage(e.getMessage());
        }
        return result;
    }

    private List<Integer> calculateInsertionPositions(int targetPageCount, InsertionOptions options) {
        ArrayList<Integer> positions = new ArrayList<Integer>();
        switch (options.getMode().ordinal()) {
            case 0: {
                positions.add(targetPageCount);
                break;
            }
            case 1: {
                int afterPage = options.getPageNumber();
                if (afterPage <= targetPageCount) {
                    positions.add(afterPage);
                    break;
                }
                if (!options.isInsertAtEndIfShorter()) break;
                positions.add(targetPageCount);
                break;
            }
            case 2: {
                int interval;
                for (int i = interval = options.getInterval(); i <= targetPageCount; i += interval) {
                    positions.add(i);
                }
                if (targetPageCount % interval == 0 || !options.isInsertAtEndIfShorter()) break;
                positions.add(targetPageCount);
            }
        }
        Collections.sort(positions);
        logger.debug("Calculated insertion positions for {} pages with mode {}: {}", new Object[]{targetPageCount, options.getMode(), positions});
        return positions;
    }

    private File generateOutputFile(File inputFile, String suffix) {
        String name = inputFile.getName();
        int dotIndex = name.lastIndexOf(46);
        String baseName = dotIndex > 0 ? name.substring(0, dotIndex) : name;
        String extension = dotIndex > 0 ? name.substring(dotIndex) : ".pdf";
        return new File(inputFile.getParentFile(), baseName + suffix + extension);
    }

    private PDRectangle getPageSize(JoinerSection.PageSize sizeOption, BufferedImage image) {
        if (sizeOption == JoinerSection.PageSize.ORIGINAL) {
            return new PDRectangle(image.getWidth(), image.getHeight());
        }
        return new PDRectangle(sizeOption.getWidth(), sizeOption.getHeight());
    }

    private float[] calculateImagePlacement(BufferedImage image, PDRectangle pageSize, JoinerSection.FitOption fitOption) {
        float x;
        float height;
        float width;
        float pageWidth = pageSize.getWidth();
        float pageHeight = pageSize.getHeight();
        float imgWidth = image.getWidth();
        float imgHeight = image.getHeight();
        return new float[]{x, switch (fitOption) {
            case JoinerSection.FitOption.FILL_PAGE -> {
                float fillScale = Math.max(pageWidth / imgWidth, pageHeight / imgHeight);
                width = imgWidth * fillScale;
                height = imgHeight * fillScale;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
            case JoinerSection.FitOption.ORIGINAL_SIZE -> {
                width = imgWidth;
                height = imgHeight;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
            default -> {
                float fitScale = Math.min(pageWidth / imgWidth, pageHeight / imgHeight);
                width = imgWidth * fitScale;
                height = imgHeight * fitScale;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
        }, width, height};
    }

    public static class InsertionOptions {
        private InsertionMode mode = InsertionMode.AT_END;
        private int pageNumber = 1;
        private int interval = 10;
        private boolean insertAtEndIfShorter = true;
        private String outputSuffix = "_modified";
        private JoinerSection.PageSize pageSize = JoinerSection.PageSize.A4;
        private JoinerSection.FitOption fitOption = JoinerSection.FitOption.FIT_TO_PAGE;

        public InsertionMode getMode() {
            return this.mode;
        }

        public void setMode(InsertionMode mode) {
            this.mode = mode;
        }

        public int getPageNumber() {
            return this.pageNumber;
        }

        public void setPageNumber(int pageNumber) {
            this.pageNumber = pageNumber;
        }

        public int getInterval() {
            return this.interval;
        }

        public void setInterval(int interval) {
            this.interval = interval;
        }

        public boolean isInsertAtEndIfShorter() {
            return this.insertAtEndIfShorter;
        }

        public void setInsertAtEndIfShorter(boolean insertAtEndIfShorter) {
            this.insertAtEndIfShorter = insertAtEndIfShorter;
        }

        public String getOutputSuffix() {
            return this.outputSuffix;
        }

        public void setOutputSuffix(String outputSuffix) {
            this.outputSuffix = outputSuffix;
        }

        public JoinerSection.PageSize getPageSize() {
            return this.pageSize;
        }

        public void setPageSize(JoinerSection.PageSize pageSize) {
            this.pageSize = pageSize;
        }

        public JoinerSection.FitOption getFitOption() {
            return this.fitOption;
        }

        public void setFitOption(JoinerSection.FitOption fitOption) {
            this.fitOption = fitOption;
        }
    }

    public static class InsertResult {
        private final File inputFile;
        private File outputFile;
        private int originalPageCount;
        private int newPageCount;
        private int insertionsPerformed;
        private boolean success;
        private String errorMessage;

        public InsertResult(File inputFile) {
            this.inputFile = inputFile;
        }

        public File getInputFile() {
            return this.inputFile;
        }

        public File getOutputFile() {
            return this.outputFile;
        }

        public void setOutputFile(File outputFile) {
            this.outputFile = outputFile;
        }

        public int getOriginalPageCount() {
            return this.originalPageCount;
        }

        public void setOriginalPageCount(int originalPageCount) {
            this.originalPageCount = originalPageCount;
        }

        public int getNewPageCount() {
            return this.newPageCount;
        }

        public void setNewPageCount(int newPageCount) {
            this.newPageCount = newPageCount;
        }

        public int getInsertionsPerformed() {
            return this.insertionsPerformed;
        }

        public void setInsertionsPerformed(int insertionsPerformed) {
            this.insertionsPerformed = insertionsPerformed;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public void setSuccess(boolean success) {
            this.success = success;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }
    }

    public static enum InsertionMode {
        AT_END,
        AFTER_PAGE,
        EVERY_N_PAGES;

    }
}

