/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.service;

import com.datmt.pdftools.model.PdfBookmark;
import com.datmt.pdftools.model.PdfDocument;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfExtractor {
    private static final Logger logger = LoggerFactory.getLogger(PdfExtractor.class);

    public void extractPages(PdfDocument sourceDocument, Set<Integer> pageIndices, File outputFile) throws IOException {
        this.extractPages(sourceDocument, pageIndices, new HashMap<Integer, Integer>(), outputFile);
    }

    public void extractPages(PdfDocument sourceDocument, Set<Integer> pageIndices, Map<Integer, Integer> rotations, File outputFile) throws IOException {
        logger.debug("Extracting pages {} from document with {} rotations", (Object)pageIndices, (Object)rotations.size());
        logger.trace("Output file: {}", (Object)outputFile.getAbsolutePath());
        PDDocument sourcePdf = sourceDocument.getPdfDocument();
        int pageCount = sourceDocument.getPageCount();
        for (int index : pageIndices) {
            if (index >= 0 && index < pageCount) continue;
            logger.error("Invalid page index: {} (document has {} pages)", (Object)index, (Object)pageCount);
            throw new IllegalArgumentException("Page index out of range: " + index);
        }
        logger.info("Creating new PDF with {} pages from {} selected pages", (Object)pageIndices.size(), (Object)pageCount);
        try (PDDocument newDocument = new PDDocument();){
            List sortedIndices = pageIndices.stream().sorted().toList();
            for (Integer pageIndex : sortedIndices) {
                logger.trace("Adding page {} to new document", (Object)pageIndex);
                PDPage page = sourcePdf.getPage(pageIndex);
                int rotation = rotations.getOrDefault(pageIndex, 0);
                if (rotation != 0) {
                    int currentRotation = page.getRotation();
                    int newRotation = (currentRotation + rotation) % 360;
                    page.setRotation(newRotation);
                    logger.debug("Page {} rotation: {} + {} = {}", pageIndex, currentRotation, rotation, newRotation);
                }
                newDocument.addPage(page);
            }
            newDocument.save(outputFile);
            logger.info("Successfully extracted {} pages to: {}", (Object)pageIndices.size(), (Object)outputFile.getAbsolutePath());
        }
        catch (IOException e) {
            logger.error("Failed to extract pages to file: {}", (Object)outputFile.getAbsolutePath(), (Object)e);
            throw e;
        }
    }

    public void extractSinglePage(PdfDocument sourceDocument, int pageIndex, File outputFile) throws IOException {
        logger.debug("Extracting single page: {} from document", (Object)pageIndex);
        TreeSet<Integer> pages = new TreeSet<Integer>();
        pages.add(pageIndex);
        this.extractPages(sourceDocument, pages, outputFile);
    }

    public void extractPageRange(PdfDocument sourceDocument, int startPage, int endPage, File outputFile) throws IOException {
        logger.debug("Extracting page range: {} to {}", (Object)startPage, (Object)endPage);
        TreeSet<Integer> pages = new TreeSet<Integer>();
        for (int i = startPage; i <= endPage; ++i) {
            pages.add(i);
        }
        this.extractPages(sourceDocument, pages, outputFile);
    }

    public void extractByBookmarks(PdfDocument sourceDocument, List<PdfBookmark> bookmarks, File outputDir) throws IOException {
        logger.info("Extracting {} bookmarks to directory: {}", (Object)bookmarks.size(), (Object)outputDir.getAbsolutePath());
        if (!outputDir.exists() && !outputDir.mkdirs()) {
            throw new IOException("Failed to create output directory: " + outputDir.getAbsolutePath());
        }
        String baseName = sourceDocument.getSourceFile().getName();
        if (baseName.toLowerCase().endsWith(".pdf")) {
            baseName = baseName.substring(0, baseName.length() - 4);
        }
        int bookmarkIndex = 1;
        for (PdfBookmark bookmark : bookmarks) {
            Object sanitizedTitle = bookmark.getSanitizedTitle();
            if (((String)sanitizedTitle).isEmpty()) {
                sanitizedTitle = "Chapter_" + bookmarkIndex;
            }
            String fileName = String.format("%s_%02d_%s.pdf", baseName, bookmarkIndex, sanitizedTitle);
            File outputFile = new File(outputDir, fileName);
            logger.debug("Extracting bookmark '{}' (pages {}-{}) to {}", bookmark.getTitle(), bookmark.getPageIndex() + 1, bookmark.getEndPageIndex() + 1, outputFile.getName());
            this.extractPageRange(sourceDocument, bookmark.getPageIndex(), bookmark.getEndPageIndex(), outputFile);
            ++bookmarkIndex;
        }
        logger.info("Successfully extracted {} bookmarks", (Object)bookmarks.size());
    }
}

