/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.service;

import com.datmt.pdftools.model.JoinerFile;
import com.datmt.pdftools.model.JoinerSection;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.List;
import javafx.embed.swing.SwingFXUtils;
import javafx.scene.image.Image;
import javax.imageio.ImageIO;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.io.RandomAccessReadBufferedFile;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.image.LosslessFactory;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfMergeService {
    private static final Logger logger = LoggerFactory.getLogger(PdfMergeService.class);
    private static final int THUMBNAIL_DPI = 72;
    private static final int PREVIEW_DPI = 150;

    public JoinerFile loadPdfFile(File file) throws IOException {
        logger.info("Loading PDF file: {}", (Object)file.getName());
        PDDocument document = Loader.loadPDF(new RandomAccessReadBufferedFile(file));
        return new JoinerFile(file, document);
    }

    public JoinerFile loadImageFile(File file) throws IOException {
        logger.info("Loading image file: {}", (Object)file.getName());
        Image image = new Image(file.toURI().toString());
        if (image.isError()) {
            throw new IOException("Failed to load image: " + file.getName());
        }
        return new JoinerFile(file, image);
    }

    public Image renderPdfThumbnail(JoinerFile joinerFile, int pageIndex) throws IOException {
        if (!joinerFile.isPdf()) {
            throw new IllegalArgumentException("File is not a PDF");
        }
        logger.trace("Rendering PDF thumbnail for page {}", (Object)pageIndex);
        PDFRenderer renderer = new PDFRenderer(joinerFile.getPdfDocument());
        BufferedImage bufferedImage = renderer.renderImageWithDPI(pageIndex, 72.0f, ImageType.RGB);
        return SwingFXUtils.toFXImage(bufferedImage, null);
    }

    public Image renderPdfPreview(JoinerFile joinerFile, int pageIndex) throws IOException {
        if (!joinerFile.isPdf()) {
            throw new IllegalArgumentException("File is not a PDF");
        }
        logger.trace("Rendering PDF preview for page {}", (Object)pageIndex);
        PDFRenderer renderer = new PDFRenderer(joinerFile.getPdfDocument());
        BufferedImage bufferedImage = renderer.renderImageWithDPI(pageIndex, 150.0f, ImageType.RGB);
        return SwingFXUtils.toFXImage(bufferedImage, null);
    }

    public Image createImageThumbnail(JoinerFile joinerFile, int maxSize) {
        if (!joinerFile.isImage()) {
            throw new IllegalArgumentException("File is not an image");
        }
        Image original = joinerFile.getImage();
        double scale = Math.min((double)maxSize / original.getWidth(), (double)maxSize / original.getHeight());
        if (scale >= 1.0) {
            return original;
        }
        return original;
    }

    public void mergeSections(List<JoinerSection> sections, File outputFile, ProgressCallback callback) throws IOException {
        logger.info("Merging {} sections to: {}", (Object)sections.size(), (Object)outputFile.getAbsolutePath());
        int totalPages = sections.stream().mapToInt(JoinerSection::getPageCount).sum();
        int processedPages = 0;
        try (PDDocument outputDoc = new PDDocument();){
            for (JoinerSection section : sections) {
                JoinerFile sourceFile = section.getSourceFile();
                logger.debug("Processing section from: {} (pages {}-{})", sourceFile.getFileName(), section.getStartPage() + 1, section.getEndPage() + 1);
                if (sourceFile.isPdf()) {
                    PDDocument sourcePdf = sourceFile.getPdfDocument();
                    int sectionRotation = section.getRotation().getDegrees();
                    for (int i = section.getStartPage(); i <= section.getEndPage(); ++i) {
                        PDPage page = sourcePdf.getPage(i);
                        PDPage importedPage = outputDoc.importPage(page);
                        if (sectionRotation != 0) {
                            int currentRotation = importedPage.getRotation();
                            int newRotation = (currentRotation + sectionRotation) % 360;
                            importedPage.setRotation(newRotation);
                            logger.debug("Applied rotation {} to page {} (total: {})", sectionRotation, i, newRotation);
                        }
                        ++processedPages;
                        if (callback == null) continue;
                        callback.onProgress(processedPages, totalPages);
                    }
                    continue;
                }
                if (!sourceFile.isImage()) continue;
                this.addImageAsPage(outputDoc, sourceFile, section);
                ++processedPages;
                if (callback == null) continue;
                callback.onProgress(processedPages, totalPages);
            }
            outputDoc.save(outputFile);
            logger.info("Successfully merged {} pages into: {}", (Object)totalPages, (Object)outputFile.getAbsolutePath());
        }
    }

    private void addImageAsPage(PDDocument document, JoinerFile imageFile, JoinerSection section) throws IOException {
        logger.debug("Adding image as page: {}", (Object)imageFile.getFileName());
        BufferedImage originalImage = ImageIO.read(imageFile.getSourceFile());
        if (originalImage == null) {
            throw new IOException("Failed to read image: " + imageFile.getFileName());
        }
        BufferedImage processedImage = this.applyRotation(originalImage, section.getRotation());
        PDRectangle pageSize = this.getPageSize(section, processedImage);
        PDPage page = new PDPage(pageSize);
        document.addPage(page);
        float[] placement = this.calculateImagePlacement(processedImage, pageSize, section.getFitOption());
        float x = placement[0];
        float y = placement[1];
        float width = placement[2];
        float height = placement[3];
        PDImageXObject pdImage = LosslessFactory.createFromImage(document, processedImage);
        try (PDPageContentStream contentStream = new PDPageContentStream(document, page);){
            contentStream.drawImage(pdImage, x, y, width, height);
        }
    }

    private BufferedImage applyRotation(BufferedImage image, JoinerSection.Rotation rotation) {
        if (rotation == JoinerSection.Rotation.NONE) {
            return image;
        }
        double radians = Math.toRadians(rotation.getDegrees());
        double sin = Math.abs(Math.sin(radians));
        double cos = Math.abs(Math.cos(radians));
        int w = image.getWidth();
        int h = image.getHeight();
        int newW = (int)Math.floor((double)w * cos + (double)h * sin);
        int newH = (int)Math.floor((double)h * cos + (double)w * sin);
        AffineTransform transform = new AffineTransform();
        transform.translate((double)(newW - w) / 2.0, (double)(newH - h) / 2.0);
        transform.rotate(radians, (double)w / 2.0, (double)h / 2.0);
        AffineTransformOp op = new AffineTransformOp(transform, 2);
        BufferedImage rotated = new BufferedImage(newW, newH, image.getType());
        op.filter(image, rotated);
        return rotated;
    }

    private PDRectangle getPageSize(JoinerSection section, BufferedImage image) {
        JoinerSection.PageSize sizeOption = section.getPageSize();
        if (sizeOption == JoinerSection.PageSize.ORIGINAL) {
            return new PDRectangle(image.getWidth(), image.getHeight());
        }
        return new PDRectangle(sizeOption.getWidth(), sizeOption.getHeight());
    }

    private float[] calculateImagePlacement(BufferedImage image, PDRectangle pageSize, JoinerSection.FitOption fitOption) {
        float x;
        float height;
        float width;
        float pageWidth = pageSize.getWidth();
        float pageHeight = pageSize.getHeight();
        float imgWidth = image.getWidth();
        float imgHeight = image.getHeight();
        return new float[]{x, switch (fitOption) {
            case JoinerSection.FitOption.FILL_PAGE -> {
                float fillScale = Math.max(pageWidth / imgWidth, pageHeight / imgHeight);
                width = imgWidth * fillScale;
                height = imgHeight * fillScale;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
            case JoinerSection.FitOption.ORIGINAL_SIZE -> {
                width = imgWidth;
                height = imgHeight;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
            default -> {
                float fitScale = Math.min(pageWidth / imgWidth, pageHeight / imgHeight);
                width = imgWidth * fitScale;
                height = imgHeight * fitScale;
                x = (pageWidth - width) / 2.0f;
                yield (pageHeight - height) / 2.0f;
            }
        }, width, height};
    }

    @FunctionalInterface
    public static interface ProgressCallback {
        public void onProgress(int var1, int var2);
    }
}

