/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.ui.compressor;

import com.datmt.pdftools.service.PdfCompressor;
import com.datmt.pdftools.util.CreditLinkHandler;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import javafx.application.Platform;
import javafx.fxml.FXML;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.CheckBox;
import javafx.scene.control.Hyperlink;
import javafx.scene.control.Label;
import javafx.scene.control.ProgressBar;
import javafx.scene.control.RadioButton;
import javafx.scene.control.Slider;
import javafx.scene.control.ToggleGroup;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.stage.FileChooser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfCompressorController {
    private static final Logger logger = LoggerFactory.getLogger(PdfCompressorController.class);
    @FXML
    private Button addFilesButton;
    @FXML
    private Button removeFileButton;
    @FXML
    private Button clearAllButton;
    @FXML
    private Button compressButton;
    @FXML
    private VBox filesContainer;
    @FXML
    private Label totalSizeLabel;
    @FXML
    private RadioButton lowCompressionRadio;
    @FXML
    private RadioButton mediumCompressionRadio;
    @FXML
    private RadioButton highCompressionRadio;
    @FXML
    private ToggleGroup compressionLevelGroup;
    @FXML
    private Slider imageQualitySlider;
    @FXML
    private Label imageQualityLabel;
    @FXML
    private CheckBox removeMetadataCheck;
    @FXML
    private CheckBox removeBookmarksCheck;
    @FXML
    private CheckBox addSuffixCheck;
    @FXML
    private CheckBox overwriteCheck;
    @FXML
    private VBox progressSection;
    @FXML
    private Label progressLabel;
    @FXML
    private ProgressBar progressBar;
    @FXML
    private Label progressDetailLabel;
    @FXML
    private VBox resultsSection;
    @FXML
    private VBox resultsContainer;
    @FXML
    private Label totalOriginalLabel;
    @FXML
    private Label totalCompressedLabel;
    @FXML
    private Label totalSavingsLabel;
    @FXML
    private Hyperlink creditLink;
    private final List<FileItem> fileItems = new ArrayList<FileItem>();
    private FileItem selectedFileItem = null;
    private final PdfCompressor compressor = new PdfCompressor();
    private final ExecutorService compressionExecutor = Executors.newSingleThreadExecutor();
    private boolean isCompressing = false;

    @FXML
    public void initialize() {
        logger.info("Initializing PDF Compressor");
        this.setupWindowCloseHandler();
        this.setupListeners();
        CreditLinkHandler.setup(this.creditLink);
        this.updateUI();
    }

    private void setupWindowCloseHandler() {
        Platform.runLater(() -> {
            if (this.addFilesButton.getScene() != null && this.addFilesButton.getScene().getWindow() != null) {
                this.addFilesButton.getScene().getWindow().setOnCloseRequest(event -> this.cleanup());
            }
        });
    }

    private void setupListeners() {
        this.compressionLevelGroup.selectedToggleProperty().addListener((obs, oldVal, newVal) -> {
            if (newVal == this.lowCompressionRadio) {
                this.imageQualitySlider.setValue(85.0);
            } else if (newVal == this.mediumCompressionRadio) {
                this.imageQualitySlider.setValue(65.0);
            } else if (newVal == this.highCompressionRadio) {
                this.imageQualitySlider.setValue(45.0);
            }
        });
        this.imageQualitySlider.valueProperty().addListener((obs, oldVal, newVal) -> this.imageQualityLabel.setText(String.format("%.0f%%", newVal.doubleValue())));
        this.overwriteCheck.selectedProperty().addListener((obs, oldVal, newVal) -> {
            if (newVal.booleanValue()) {
                this.addSuffixCheck.setSelected(false);
            }
        });
        this.addSuffixCheck.selectedProperty().addListener((obs, oldVal, newVal) -> {
            if (newVal.booleanValue()) {
                this.overwriteCheck.setSelected(false);
            }
        });
    }

    @FXML
    private void onAddFiles() {
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select PDF Files");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        List<File> files = fileChooser.showOpenMultipleDialog(this.addFilesButton.getScene().getWindow());
        if (files != null) {
            for (File file : files) {
                this.addFile(file);
            }
            this.updateUI();
        }
    }

    private void addFile(File file) {
        for (FileItem item : this.fileItems) {
            if (!item.getFile().getAbsolutePath().equals(file.getAbsolutePath())) continue;
            logger.debug("File already added: {}", (Object)file.getName());
            return;
        }
        FileItem item = new FileItem(file, this::onFileSelected, this::onFileRemove);
        this.fileItems.add(item);
        this.filesContainer.getChildren().add(item.getNode());
        logger.info("Added file: {} ({})", (Object)file.getName(), (Object)PdfCompressor.formatFileSize(file.length()));
    }

    private void onFileSelected(FileItem item) {
        if (this.selectedFileItem != null) {
            this.selectedFileItem.setSelected(false);
        }
        this.selectedFileItem = item;
        item.setSelected(true);
        this.updateUI();
    }

    private void onFileRemove(FileItem item) {
        this.fileItems.remove(item);
        this.filesContainer.getChildren().remove(item.getNode());
        if (this.selectedFileItem == item) {
            this.selectedFileItem = null;
        }
        this.updateUI();
    }

    @FXML
    private void onRemoveFile() {
        if (this.selectedFileItem != null) {
            this.onFileRemove(this.selectedFileItem);
        }
    }

    @FXML
    private void onClearAll() {
        this.fileItems.clear();
        this.filesContainer.getChildren().clear();
        this.selectedFileItem = null;
        this.resultsSection.setVisible(false);
        this.resultsSection.setManaged(false);
        this.resultsContainer.getChildren().clear();
        this.updateUI();
    }

    @FXML
    private void onCompress() {
        if (this.fileItems.isEmpty() || this.isCompressing) {
            return;
        }
        PdfCompressor.CompressionOptions options = new PdfCompressor.CompressionOptions();
        if (this.lowCompressionRadio.isSelected()) {
            options.setLevel(PdfCompressor.CompressionLevel.LOW);
        } else if (this.highCompressionRadio.isSelected()) {
            options.setLevel(PdfCompressor.CompressionLevel.HIGH);
        } else {
            options.setLevel(PdfCompressor.CompressionLevel.MEDIUM);
        }
        options.setImageQuality((float)this.imageQualitySlider.getValue() / 100.0f);
        options.setRemoveMetadata(this.removeMetadataCheck.isSelected());
        options.setRemoveBookmarks(this.removeBookmarksCheck.isSelected());
        boolean addSuffix = this.addSuffixCheck.isSelected();
        boolean overwrite = this.overwriteCheck.isSelected();
        this.isCompressing = true;
        this.updateUI();
        this.progressSection.setVisible(true);
        this.progressSection.setManaged(true);
        this.progressBar.setProgress(0.0);
        this.resultsSection.setVisible(false);
        this.resultsSection.setManaged(false);
        this.resultsContainer.getChildren().clear();
        ArrayList<File> filesToCompress = new ArrayList<File>();
        for (FileItem item : this.fileItems) {
            filesToCompress.add(item.getFile());
        }
        int totalFiles = filesToCompress.size();
        this.compressionExecutor.submit(() -> {
            ArrayList<PdfCompressor.CompressResult> allResults = new ArrayList<PdfCompressor.CompressResult>();
            for (int i = 0; i < totalFiles; ++i) {
                File inputFile = (File)filesToCompress.get(i);
                int fileIndex = i;
                Platform.runLater(() -> {
                    this.progressLabel.setText("Compressing: " + inputFile.getName() + " (" + (fileIndex + 1) + "/" + totalFiles + ")");
                    this.progressBar.setProgress((double)fileIndex / (double)totalFiles);
                });
                logger.info("Starting compression of: {}", (Object)inputFile.getName());
                File outputFile = overwrite ? new File(inputFile.getParent(), inputFile.getName().replace(".pdf", "_temp_compress.pdf")) : new File(inputFile.getParent(), inputFile.getName().replace(".pdf", "_compressed.pdf"));
                PdfCompressor.CompressResult result = this.compressor.compress(inputFile, outputFile, options, (page, total, status) -> Platform.runLater(() -> this.progressDetailLabel.setText(status + " - page " + page + "/" + total)));
                if (overwrite && result.isSuccess()) {
                    try {
                        if (inputFile.delete() && outputFile.renameTo(inputFile)) {
                            result = new PdfCompressor.CompressResult(inputFile, inputFile, result.getOriginalSize(), result.getCompressedSize());
                        }
                    }
                    catch (Exception e) {
                        logger.error("Failed to replace original file: {}", (Object)e.getMessage());
                    }
                }
                allResults.add(result);
                logger.info("Finished compression of: {} (success={})", (Object)inputFile.getName(), (Object)result.isSuccess());
            }
            Platform.runLater(() -> {
                this.showResults(allResults);
                this.isCompressing = false;
                this.progressSection.setVisible(false);
                this.progressSection.setManaged(false);
                this.updateUI();
            });
        });
    }

    private void showResults(List<PdfCompressor.CompressResult> results) {
        this.resultsContainer.getChildren().clear();
        long totalOriginal = 0L;
        long totalCompressed = 0L;
        int successCount = 0;
        for (PdfCompressor.CompressResult result : results) {
            HBox row = new HBox(10.0);
            row.setAlignment(Pos.CENTER_LEFT);
            Label nameLabel = new Label(result.getInputFile().getName());
            nameLabel.setMaxWidth(200.0);
            nameLabel.setStyle("-fx-font-size: 11;");
            if (result.isSuccess()) {
                ++successCount;
                totalOriginal += result.getOriginalSize();
                totalCompressed += result.getCompressedSize();
                String sizeText = String.format("%s -> %s (-%,.0f%%)", PdfCompressor.formatFileSize(result.getOriginalSize()), PdfCompressor.formatFileSize(result.getCompressedSize()), result.getCompressionRatio() * 100.0);
                Label sizeLabel = new Label(sizeText);
                sizeLabel.setStyle("-fx-font-size: 11; -fx-text-fill: #4CAF50;");
                row.getChildren().addAll((Node[])new Node[]{nameLabel, sizeLabel});
            } else {
                Label errorLabel = new Label("Failed: " + result.getErrorMessage());
                errorLabel.setStyle("-fx-font-size: 11; -fx-text-fill: #cc0000;");
                row.getChildren().addAll((Node[])new Node[]{nameLabel, errorLabel});
            }
            this.resultsContainer.getChildren().add(row);
        }
        this.totalOriginalLabel.setText("Original: " + PdfCompressor.formatFileSize(totalOriginal));
        this.totalCompressedLabel.setText("Compressed: " + PdfCompressor.formatFileSize(totalCompressed));
        if (totalOriginal > 0L) {
            double savings = (1.0 - (double)totalCompressed / (double)totalOriginal) * 100.0;
            this.totalSavingsLabel.setText(String.format("Saved: %.1f%%", savings));
        } else {
            this.totalSavingsLabel.setText("Saved: 0%");
        }
        this.resultsSection.setVisible(true);
        this.resultsSection.setManaged(true);
        logger.info("Compression complete: {}/{} files successful", (Object)successCount, (Object)results.size());
    }

    private void updateUI() {
        boolean hasFiles = !this.fileItems.isEmpty();
        boolean hasSelection = this.selectedFileItem != null;
        this.removeFileButton.setDisable(!hasSelection || this.isCompressing);
        this.clearAllButton.setDisable(!hasFiles || this.isCompressing);
        this.compressButton.setDisable(!hasFiles || this.isCompressing);
        this.addFilesButton.setDisable(this.isCompressing);
        long totalSize = 0L;
        for (FileItem item : this.fileItems) {
            totalSize += item.getFile().length();
        }
        this.totalSizeLabel.setText(String.format("Total: %d files, %s", this.fileItems.size(), PdfCompressor.formatFileSize(totalSize)));
    }

    public void cleanup() {
        logger.info("Cleaning up PDF Compressor");
        this.compressionExecutor.shutdownNow();
        this.fileItems.clear();
        this.filesContainer.getChildren().clear();
    }

    private static class FileItem {
        private final File file;
        private final HBox node;
        private boolean selected = false;

        public FileItem(File file, Consumer<FileItem> onSelect, Consumer<FileItem> onRemove) {
            this.file = file;
            this.node = this.createNode(onSelect, onRemove);
        }

        private HBox createNode(Consumer<FileItem> onSelect, Consumer<FileItem> onRemove) {
            HBox box = new HBox(10.0);
            box.setAlignment(Pos.CENTER_LEFT);
            box.setStyle("-fx-padding: 8; -fx-background-color: white; -fx-border-color: #e0e0e0; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            Label nameLabel = new Label(this.file.getName());
            nameLabel.setStyle("-fx-font-size: 13;");
            nameLabel.setMaxWidth(250.0);
            HBox.setHgrow(nameLabel, Priority.ALWAYS);
            Label sizeLabel = new Label(PdfCompressor.formatFileSize(this.file.length()));
            sizeLabel.setStyle("-fx-font-size: 12; -fx-text-fill: #666666;");
            Button removeBtn = new Button("X");
            removeBtn.setStyle("-fx-font-size: 10; -fx-padding: 2 6; -fx-background-color: #ff5252; -fx-text-fill: white; -fx-cursor: hand;");
            removeBtn.setOnAction(e -> onRemove.accept(this));
            box.getChildren().addAll((Node[])new Node[]{nameLabel, sizeLabel, removeBtn});
            box.setOnMouseClicked(e -> onSelect.accept(this));
            return box;
        }

        public File getFile() {
            return this.file;
        }

        public HBox getNode() {
            return this.node;
        }

        public void setSelected(boolean selected) {
            this.selected = selected;
            if (selected) {
                this.node.setStyle("-fx-padding: 8; -fx-background-color: #e3f2fd; -fx-border-color: #2196F3; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            } else {
                this.node.setStyle("-fx-padding: 8; -fx-background-color: white; -fx-border-color: #e0e0e0; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            }
        }
    }
}

