/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.ui.extractor;

import com.datmt.pdftools.model.PdfBookmark;
import com.datmt.pdftools.model.PdfDocument;
import com.datmt.pdftools.service.PdfService;
import com.datmt.pdftools.ui.extractor.components.PageThumbnailPanel;
import com.datmt.pdftools.util.CreditLinkHandler;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javafx.animation.KeyFrame;
import javafx.animation.KeyValue;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.concurrent.Task;
import javafx.fxml.FXML;
import javafx.geometry.Bounds;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.control.Button;
import javafx.scene.control.CheckBox;
import javafx.scene.control.Hyperlink;
import javafx.scene.control.Label;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.Spinner;
import javafx.scene.control.SpinnerValueFactory;
import javafx.scene.control.Tab;
import javafx.scene.control.TabPane;
import javafx.scene.control.TextField;
import javafx.scene.control.TreeCell;
import javafx.scene.control.TreeItem;
import javafx.scene.control.TreeView;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.Pane;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.TextAlignment;
import javafx.stage.DirectoryChooser;
import javafx.stage.FileChooser;
import javafx.stage.Stage;
import javafx.util.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfExtractorController {
    private static final Logger logger = LoggerFactory.getLogger(PdfExtractorController.class);
    private static final int MAX_RENDERED_THUMBNAILS = 100;
    private static final int BUFFER_SIZE = 5;
    private static final int PANEL_HEIGHT = 130;
    Task<Void> currentLoadingTask;
    @FXML
    private Button loadButton;
    @FXML
    private Label fileLabel;
    @FXML
    private Label pageCountLabel;
    @FXML
    private VBox pagesListContainer;
    @FXML
    private ScrollPane pagesScrollPane;
    @FXML
    private StackPane previewContainer;
    @FXML
    private ScrollPane previewScrollPane;
    @FXML
    private Label currentPageLabel;
    @FXML
    private Button prevButton;
    @FXML
    private Button nextButton;
    @FXML
    private VBox selectedPagesList;
    @FXML
    private ScrollPane selectedPagesScrollPane;
    @FXML
    private TextField pageInputField;
    @FXML
    private Button addPagesButton;
    @FXML
    private Label inputHintLabel;
    @FXML
    private Button removeSelectedButton;
    @FXML
    private Button clearButton;
    @FXML
    private TextField outputFileField;
    @FXML
    private Button browseButton;
    @FXML
    private Button exportButton;
    @FXML
    private Button selectAllButton;
    @FXML
    private Button deselectAllButton;
    @FXML
    private Label notificationLabel;
    @FXML
    private TabPane leftTabPane;
    @FXML
    private Tab pagesTab;
    @FXML
    private Tab bookmarksTab;
    @FXML
    private TreeView<PdfBookmark> bookmarkTreeView;
    @FXML
    private Button selectAllBookmarksButton;
    @FXML
    private Button deselectAllBookmarksButton;
    @FXML
    private Spinner<Integer> levelSpinner;
    @FXML
    private Button selectLevelButton;
    @FXML
    private TextField outputDirField;
    @FXML
    private Button browseDirButton;
    @FXML
    private Button exportBookmarksButton;
    @FXML
    private Button rotateLeftButton;
    @FXML
    private Button rotateRightButton;
    @FXML
    private Hyperlink creditLink;
    private PdfService pdfService;
    private Set<Integer> selectedPages;
    private List<PageThumbnailPanel> thumbnailPanels;
    private int currentPreviewPage;
    private ImageView currentImageView;
    private PageThumbnailPanel currentSelectedPanel;
    private List<PdfBookmark> currentBookmarks;
    private final ExecutorService renderExecutor = Executors.newFixedThreadPool(4);
    private final AtomicReference<Object> currentSessionId = new AtomicReference();
    private Image placeholderImage;
    private Set<Integer> renderedThumbnails = new HashSet<Integer>();
    private Timeline scrollDebounceTimeline;
    private Map<Integer, Integer> pageRotations = new HashMap<Integer, Integer>();
    private boolean isBulkOperation = false;

    @FXML
    public void initialize() {
        logger.trace("Initializing PdfExtractorController");
        this.pdfService = new PdfService();
        this.selectedPages = new TreeSet<Integer>();
        this.thumbnailPanels = new ArrayList<PageThumbnailPanel>();
        this.currentBookmarks = new ArrayList<PdfBookmark>();
        this.currentPreviewPage = 0;
        CreditLinkHandler.setup(this.creditLink);
        this.setupEventHandlers();
        this.setupBookmarkTreeView();
        this.setupLazyLoading();
        this.setupWindowCloseHandler();
        logger.debug("Controller initialization complete");
    }

    private void setupWindowCloseHandler() {
        Platform.runLater(() -> {
            if (this.loadButton.getScene() != null && this.loadButton.getScene().getWindow() != null) {
                this.loadButton.getScene().getWindow().setOnCloseRequest(event -> this.cleanup());
            }
        });
    }

    public void cleanup() {
        logger.info("Cleaning up PdfExtractorController resources");
        this.renderExecutor.shutdownNow();
        this.pdfService.closeDocument();
        this.thumbnailPanels.clear();
        this.renderedThumbnails.clear();
        this.pagesListContainer.getChildren().clear();
        this.previewContainer.getChildren().clear();
        if (this.scrollDebounceTimeline != null) {
            this.scrollDebounceTimeline.stop();
        }
        logger.info("Cleanup complete");
    }

    private void setupLazyLoading() {
        this.placeholderImage = this.createPlaceholderImage();
        this.scrollDebounceTimeline = new Timeline(new KeyFrame(Duration.millis(150.0), e -> this.updateVisibleThumbnails(), new KeyValue[0]));
        this.scrollDebounceTimeline.setCycleCount(1);
        this.pagesScrollPane.vvalueProperty().addListener((obs, oldVal, newVal) -> {
            this.scrollDebounceTimeline.stop();
            this.scrollDebounceTimeline.playFromStart();
        });
        this.pagesScrollPane.viewportBoundsProperty().addListener((obs, oldVal, newVal) -> {
            this.scrollDebounceTimeline.stop();
            this.scrollDebounceTimeline.playFromStart();
        });
    }

    private Image createPlaceholderImage() {
        int size = 100;
        Canvas canvas = new Canvas(size, size);
        GraphicsContext gc = canvas.getGraphicsContext2D();
        gc.setFill(Color.rgb(230, 230, 230));
        gc.fillRect(0.0, 0.0, size, size);
        gc.setStroke(Color.rgb(200, 200, 200));
        gc.strokeRect(0.0, 0.0, size, size);
        gc.setFill(Color.rgb(150, 150, 150));
        gc.setFont(Font.font(12.0));
        gc.setTextAlign(TextAlignment.CENTER);
        gc.fillText("Loading...", size / 2, size / 2);
        return canvas.snapshot(null, null);
    }

    private void setupBookmarkTreeView() {
        this.bookmarkTreeView.setCellFactory(tv -> new CheckBoxTreeCell<PdfBookmark>(){

            @Override
            public void updateItem(PdfBookmark item, boolean empty) {
                super.updateItem(item, empty);
                if (empty || item == null) {
                    this.setText(null);
                    this.setGraphic(null);
                } else {
                    CheckBox checkBox = new CheckBox();
                    checkBox.setSelected(item.isSelected());
                    checkBox.setOnAction(e -> {
                        item.setSelected(checkBox.isSelected());
                        PdfExtractorController.this.updateBookmarkExportButton();
                    });
                    String displayText = item.getTitle() + " (" + item.getPageRangeString() + ")";
                    this.setText(displayText);
                    this.setGraphic(checkBox);
                }
            }
        });
        this.bookmarkTreeView.setOnMouseClicked(event -> {
            TreeItem selectedItem = (TreeItem)this.bookmarkTreeView.getSelectionModel().getSelectedItem();
            if (selectedItem != null && selectedItem.getValue() != null) {
                PdfBookmark bookmark = (PdfBookmark)selectedItem.getValue();
                int pageIndex = bookmark.getPageIndex();
                logger.debug("Bookmark clicked: '{}', jumping to page {}", (Object)bookmark.getTitle(), (Object)(pageIndex + 1));
                this.jumpToPage(pageIndex);
            }
        });
    }

    private void jumpToPage(int pageIndex) {
        if (!this.pdfService.isDocumentLoaded() || pageIndex < 0 || pageIndex >= this.thumbnailPanels.size()) {
            return;
        }
        this.updatePreview(pageIndex);
        this.scrollToThumbnail(pageIndex);
    }

    private void scrollToThumbnail(int pageIndex) {
        double viewportHeight;
        if (this.thumbnailPanels.isEmpty() || pageIndex < 0 || pageIndex >= this.thumbnailPanels.size()) {
            return;
        }
        double contentHeight = this.pagesListContainer.getHeight();
        if (contentHeight <= (viewportHeight = this.pagesScrollPane.getViewportBounds().getHeight())) {
            return;
        }
        PageThumbnailPanel targetPanel = this.thumbnailPanels.get(pageIndex);
        double panelY = targetPanel.getBoundsInParent().getMinY();
        double scrollY = panelY - viewportHeight / 2.0 + 65.0;
        double scrollValue = Math.max(0.0, Math.min(1.0, scrollY / (contentHeight - viewportHeight)));
        this.pagesScrollPane.setVvalue(scrollValue);
    }

    private void setupEventHandlers() {
        logger.trace("Setting up event handlers");
        this.pageInputField.setOnAction(event -> this.onAddPages());
    }

    @FXML
    private void onLoadPdf() {
        logger.info("User clicked Load PDF");
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select PDF File");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        Stage stage = (Stage)this.loadButton.getScene().getWindow();
        File selectedFile = fileChooser.showOpenDialog(stage);
        if (selectedFile != null) {
            this.loadPdfFile(selectedFile);
        }
    }

    private void loadPdfFile(final File file) {
        logger.info("Loading PDF file: {}", (Object)file.getAbsolutePath());
        Task<PdfDocument> loadTask = new Task<PdfDocument>(){

            @Override
            protected PdfDocument call() throws Exception {
                logger.debug("Background task: loading PDF");
                return PdfExtractorController.this.pdfService.loadPdf(file);
            }
        };
        loadTask.setOnSucceeded(event -> {
            PdfDocument doc = (PdfDocument)loadTask.getValue();
            logger.info("PDF loaded successfully: {} pages", (Object)doc.getPageCount());
            this.fileLabel.setText(file.getName());
            this.pageCountLabel.setText(doc.getPageCount() + " pages");
            this.selectedPages.clear();
            this.selectedPagesList.getChildren().clear();
            this.pageRotations.clear();
            this.currentPreviewPage = 0;
            this.loadPageThumbnails(doc, this.pagesListContainer, this.thumbnailPanels);
            this.updatePreview(0);
            this.removeSelectedButton.setDisable(this.selectedPagesList.getChildren().isEmpty());
            this.clearButton.setDisable(true);
            this.rotateLeftButton.setDisable(false);
            this.rotateRightButton.setDisable(false);
            this.loadBookmarks();
        });
        loadTask.setOnFailed(event -> {
            logger.error("Failed to load PDF", loadTask.getException());
            this.showError("Failed to load PDF: " + loadTask.getException().getMessage());
        });
        new Thread(loadTask).start();
    }

    public void loadPageThumbnailsParallel(PdfDocument document, Pane container, List<PageThumbnailPanel> trackerList) {
        Object mySessionId = new Object();
        this.currentSessionId.set(mySessionId);
        logger.debug("Starting parallel thumbnail generation. Session: {}", mySessionId);
        container.getChildren().clear();
        if (trackerList != null) {
            trackerList.clear();
        }
        ConcurrentHashMap resultsMap = new ConcurrentHashMap();
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        int i = 0;
        while (i < document.getPageCount()) {
            int pageIndex = i++;
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                if (this.currentSessionId.get() != mySessionId) {
                    return;
                }
                try {
                    Image thumbnail = this.pdfService.renderPageThumbnail(pageIndex);
                    PageThumbnailPanel panel = new PageThumbnailPanel(pageIndex + 1, thumbnail, selected -> this.onPageThumbnailToggled(pageIndex, (boolean)selected), clickedPage -> this.updatePreview(pageIndex));
                    resultsMap.put(pageIndex, panel);
                }
                catch (Exception e) {
                    logger.error("Failed to render page {}", (Object)pageIndex, (Object)e);
                }
            }, this.renderExecutor);
            futures.add(future);
        }
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenRunAsync(() -> {
            if (this.currentSessionId.get() != mySessionId) {
                logger.debug("Loading cancelled or superseded. Ignoring results.");
                return;
            }
            List sortedPanels = resultsMap.entrySet().stream().sorted(Map.Entry.comparingByKey()).map(Map.Entry::getValue).collect(Collectors.toList());
            Platform.runLater(() -> {
                if (this.currentSessionId.get() == mySessionId) {
                    container.getChildren().setAll(sortedPanels);
                    if (trackerList != null) {
                        trackerList.addAll(sortedPanels);
                    }
                    logger.info("Finished loading {} thumbnails.", (Object)sortedPanels.size());
                }
            });
        }, Platform::runLater);
    }

    public void loadPageThumbnails(PdfDocument document, Pane container, List<PageThumbnailPanel> trackerList) {
        container.getChildren().clear();
        if (trackerList != null) {
            trackerList.clear();
        }
        this.renderedThumbnails.clear();
        logger.debug("Creating {} placeholder panels for lazy loading", (Object)document.getPageCount());
        for (int i = 0; i < document.getPageCount(); ++i) {
            int pageIndex = i;
            PageThumbnailPanel panel = new PageThumbnailPanel(pageIndex + 1, this.placeholderImage, selected -> this.onPageThumbnailToggled(pageIndex, (boolean)selected), unused -> this.updatePreview(pageIndex));
            container.getChildren().add(panel);
            if (trackerList == null) continue;
            trackerList.add(panel);
        }
        logger.debug("Created {} placeholder panels, triggering initial lazy load", (Object)document.getPageCount());
        Platform.runLater(() -> {
            Timeline initialLoad = new Timeline(new KeyFrame(Duration.millis(100.0), e -> this.updateVisibleThumbnails(), new KeyValue[0]));
            initialLoad.play();
        });
    }

    private void updateVisibleThumbnails() {
        if (this.thumbnailPanels.isEmpty() || !this.pdfService.isDocumentLoaded()) {
            return;
        }
        int[] visibleRange = this.getVisiblePanelRange();
        int firstVisible = visibleRange[0];
        int lastVisible = visibleRange[1];
        logger.trace("Visible range: {} to {}, currently rendered: {}", firstVisible, lastVisible, this.renderedThumbnails.size());
        HashSet<Integer> toLoad = new HashSet<Integer>();
        for (int i = firstVisible; i <= lastVisible; ++i) {
            if (i < 0 || i >= this.thumbnailPanels.size() || this.renderedThumbnails.contains(i)) continue;
            toLoad.add(i);
        }
        int willBeRendered = this.renderedThumbnails.size() + toLoad.size();
        if (willBeRendered > 100) {
            int toUnload = willBeRendered - 100;
            this.unloadFurthestThumbnails(firstVisible, lastVisible, toUnload);
        }
        Iterator iterator = toLoad.iterator();
        while (iterator.hasNext()) {
            int pageIndex = (Integer)iterator.next();
            this.loadThumbnailForPage(pageIndex);
        }
    }

    private int[] getVisiblePanelRange() {
        if (this.thumbnailPanels.isEmpty()) {
            return new int[]{0, 0};
        }
        Bounds viewportBoundsInScene = this.pagesScrollPane.getViewportBounds();
        Bounds scrollPaneBoundsInScene = this.pagesScrollPane.localToScene(this.pagesScrollPane.getBoundsInLocal());
        if (scrollPaneBoundsInScene == null) {
            return new int[]{0, Math.min(10, this.thumbnailPanels.size() - 1)};
        }
        double viewportTop = scrollPaneBoundsInScene.getMinY();
        double viewportBottom = scrollPaneBoundsInScene.getMaxY();
        int firstVisible = -1;
        int lastVisible = -1;
        for (int i = 0; i < this.thumbnailPanels.size(); ++i) {
            boolean isVisible;
            PageThumbnailPanel panel = this.thumbnailPanels.get(i);
            Bounds panelBoundsInScene = panel.localToScene(panel.getBoundsInLocal());
            if (panelBoundsInScene == null) continue;
            double panelTop = panelBoundsInScene.getMinY();
            double panelBottom = panelBoundsInScene.getMaxY();
            boolean bl = isVisible = panelBottom >= viewportTop && panelTop <= viewportBottom;
            if (isVisible) {
                if (firstVisible == -1) {
                    firstVisible = i;
                }
                lastVisible = i;
                continue;
            }
            if (firstVisible != -1) break;
        }
        if (firstVisible == -1) {
            firstVisible = 0;
            lastVisible = Math.min(10, this.thumbnailPanels.size() - 1);
        }
        firstVisible = Math.max(0, firstVisible - 5);
        lastVisible = Math.min(this.thumbnailPanels.size() - 1, lastVisible + 5);
        return new int[]{firstVisible, lastVisible};
    }

    private void loadThumbnailForPage(int pageIndex) {
        if (this.renderedThumbnails.contains(pageIndex) || pageIndex < 0 || pageIndex >= this.thumbnailPanels.size()) {
            return;
        }
        this.renderedThumbnails.add(pageIndex);
        CompletableFuture.runAsync(() -> {
            try {
                Image thumbnail = this.pdfService.renderPageThumbnail(pageIndex);
                Platform.runLater(() -> {
                    if (pageIndex < this.thumbnailPanels.size()) {
                        this.thumbnailPanels.get(pageIndex).setThumbnail(thumbnail);
                        logger.trace("Loaded thumbnail for page {}", (Object)pageIndex);
                    }
                });
            }
            catch (Exception e) {
                logger.error("Failed to load thumbnail for page {}", (Object)pageIndex, (Object)e);
                this.renderedThumbnails.remove(pageIndex);
            }
        }, this.renderExecutor);
    }

    private void unloadFurthestThumbnails(int firstVisible, int lastVisible, int count) {
        ArrayList<Integer> candidates = new ArrayList<Integer>(this.renderedThumbnails);
        int center = (firstVisible + lastVisible) / 2;
        candidates.sort((a, b) -> {
            int distA = Math.min(Math.abs(a - firstVisible), Math.abs(a - lastVisible));
            int distB = Math.min(Math.abs(b - firstVisible), Math.abs(b - lastVisible));
            return Integer.compare(distB, distA);
        });
        int unloaded = 0;
        Iterator iterator = candidates.iterator();
        while (iterator.hasNext()) {
            int pageIndex = (Integer)iterator.next();
            if (unloaded >= count) break;
            if (pageIndex >= firstVisible && pageIndex <= lastVisible || pageIndex < 0 || pageIndex >= this.thumbnailPanels.size()) continue;
            this.thumbnailPanels.get(pageIndex).clearThumbnail(this.placeholderImage);
            this.renderedThumbnails.remove(pageIndex);
            ++unloaded;
            logger.trace("Unloaded thumbnail for page {}", (Object)pageIndex);
        }
    }

    private void onPageThumbnailToggled(int pageIndex, boolean selected) {
        if (this.isBulkOperation) {
            return;
        }
        logger.debug("Page {} thumbnail toggled, selected: {}", (Object)pageIndex, (Object)selected);
        if (selected) {
            this.selectedPages.add(pageIndex);
            logger.trace("Page {} added to selection", (Object)pageIndex);
        } else {
            this.selectedPages.remove(pageIndex);
            logger.trace("Page {} removed from selection", (Object)pageIndex);
        }
        this.updateSelectedPagesList();
    }

    private void updatePreview(final int pageIndex) {
        logger.debug("Updating preview for page {}", (Object)pageIndex);
        if (!this.pdfService.isDocumentLoaded()) {
            logger.warn("Cannot update preview: no document loaded");
            return;
        }
        if (this.currentSelectedPanel != null) {
            this.currentSelectedPanel.setPreviewSelected(false);
        }
        if (pageIndex >= 0 && pageIndex < this.thumbnailPanels.size()) {
            this.currentSelectedPanel = this.thumbnailPanels.get(pageIndex);
            this.currentSelectedPanel.setPreviewSelected(true);
        }
        this.currentPreviewPage = pageIndex;
        this.currentPageLabel.setText("Page " + (pageIndex + 1));
        this.prevButton.setDisable(pageIndex == 0);
        this.nextButton.setDisable(pageIndex >= this.pdfService.getCurrentDocument().getPageCount() - 1);
        Task<Image> renderTask = new Task<Image>(){

            @Override
            protected Image call() throws Exception {
                logger.trace("Background task: rendering page {} for preview", (Object)pageIndex);
                return PdfExtractorController.this.pdfService.renderPage(pageIndex);
            }
        };
        renderTask.setOnSucceeded(e -> {
            Image image = (Image)renderTask.getValue();
            if (image == null) {
                logger.warn("Rendered image is null for page {}", (Object)pageIndex);
                return;
            }
            this.previewContainer.getChildren().clear();
            this.currentImageView = new ImageView(image);
            this.currentImageView.setPreserveRatio(true);
            int rotation = this.pageRotations.getOrDefault(pageIndex, 0);
            this.currentImageView.setRotate(rotation);
            double containerWidth = this.previewContainer.getWidth();
            if (containerWidth > 20.0) {
                this.currentImageView.setFitWidth(containerWidth - 20.0);
            } else {
                this.currentImageView.fitWidthProperty().bind(this.previewContainer.widthProperty().subtract(20));
            }
            this.previewContainer.getChildren().add(this.currentImageView);
            logger.trace("Page {} preview rendered and displayed with rotation {}", (Object)pageIndex, (Object)rotation);
        });
        renderTask.setOnFailed(event -> {
            logger.error("Failed to render page preview", renderTask.getException());
            this.showError("Failed to render page preview");
        });
        new Thread(renderTask).start();
    }

    @FXML
    private void onPreviousPage() {
        if (this.currentPreviewPage > 0) {
            logger.debug("Moving to previous page from {}", (Object)this.currentPreviewPage);
            this.updatePreview(this.currentPreviewPage - 1);
        }
    }

    @FXML
    private void onNextPage() {
        int pageCount = this.pdfService.getCurrentDocument().getPageCount();
        if (this.currentPreviewPage < pageCount - 1) {
            logger.debug("Moving to next page from {}", (Object)this.currentPreviewPage);
            this.updatePreview(this.currentPreviewPage + 1);
        }
    }

    @FXML
    private void onAddPages() {
        String input = this.pageInputField.getText().trim();
        logger.info("User entered page selection: {}", (Object)input);
        if (input.isEmpty()) {
            logger.warn("Empty page input");
            this.showWarning("Please enter page numbers");
            return;
        }
        try {
            Set<Integer> pagesToAdd = this.parsePageInput(input);
            logger.debug("Parsed {} pages from input", (Object)pagesToAdd.size());
            this.selectedPages.addAll(pagesToAdd);
            this.updateSelectedPagesList();
            this.pageInputField.clear();
            logger.info("Selected pages updated: {}", (Object)this.selectedPages);
            this.showInfo("Added " + pagesToAdd.size() + " page(s)");
        }
        catch (IllegalArgumentException e) {
            logger.warn("Invalid page input: {}", (Object)e.getMessage());
            this.showWarning(e.getMessage());
        }
    }

    private Set<Integer> parsePageInput(String input) throws IllegalArgumentException {
        String[] parts;
        logger.trace("Parsing page input: {}", (Object)input);
        TreeSet<Integer> pages = new TreeSet<Integer>();
        int pageCount = this.pdfService.getCurrentDocument().getPageCount();
        for (String part : parts = input.split(",")) {
            if ((part = part.trim()).contains("-")) {
                String[] range = part.split("-");
                if (range.length != 2) {
                    throw new IllegalArgumentException("Invalid range format: " + part);
                }
                try {
                    int start = Integer.parseInt(range[0].trim()) - 1;
                    int end = Integer.parseInt(range[1].trim()) - 1;
                    if (start < 0 || end >= pageCount || start > end) {
                        throw new IllegalArgumentException("Invalid range: " + part);
                    }
                    for (int i = start; i <= end; ++i) {
                        pages.add(i);
                    }
                    logger.trace("Added range: {} to {} (0-based)", (Object)start, (Object)end);
                    continue;
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Invalid range: " + part);
                }
            }
            try {
                int pageNum = Integer.parseInt(part) - 1;
                if (pageNum < 0 || pageNum >= pageCount) {
                    throw new IllegalArgumentException("Page out of range: " + (pageNum + 1));
                }
                pages.add(pageNum);
                logger.trace("Added page: {}", (Object)pageNum);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid page number: " + part);
            }
        }
        return pages;
    }

    private void updateSelectedPagesList() {
        logger.debug("Updating selected pages list with {} pages", (Object)this.selectedPages.size());
        this.selectedPagesList.getChildren().clear();
        int count = this.selectedPages.size();
        if (count != 0) {
            if (count > 50) {
                Label summaryLabel = new Label(count + " pages selected");
                summaryLabel.setStyle("-fx-padding: 5; -fx-font-weight: bold;");
                this.selectedPagesList.getChildren().add(summaryLabel);
                String rangeStr = this.formatPageRanges(this.selectedPages);
                if (rangeStr.length() <= 100) {
                    Label rangeLabel = new Label(rangeStr);
                    rangeLabel.setStyle("-fx-padding: 5; -fx-font-size: 10; -fx-text-fill: #666;");
                    rangeLabel.setWrapText(true);
                    this.selectedPagesList.getChildren().add(rangeLabel);
                }
            } else {
                for (Integer pageNum : this.selectedPages) {
                    Label label = new Label("" + (pageNum + 1));
                    label.setStyle("-fx-padding: 5; -fx-border-color: #cccccc; -fx-border-radius: 3;");
                    this.selectedPagesList.getChildren().add(label);
                }
            }
        }
        this.removeSelectedButton.setDisable(count == 0);
        this.clearButton.setDisable(count == 0);
        this.exportButton.setDisable(count == 0);
    }

    private String formatPageRanges(Set<Integer> pages) {
        int rangeStart;
        if (pages.isEmpty()) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        List sorted = pages.stream().sorted().collect(Collectors.toList());
        int rangeEnd = rangeStart = ((Integer)sorted.get(0)).intValue();
        for (int i = 1; i < sorted.size(); ++i) {
            int current = (Integer)sorted.get(i);
            if (current == rangeEnd + 1) {
                rangeEnd = current;
                continue;
            }
            if (sb.length() > 0) {
                sb.append(", ");
            }
            sb.append(rangeStart + 1);
            if (rangeEnd > rangeStart) {
                sb.append("-").append(rangeEnd + 1);
            }
            rangeStart = current;
            rangeEnd = current;
        }
        if (sb.length() > 0) {
            sb.append(", ");
        }
        sb.append(rangeStart + 1);
        if (rangeEnd > rangeStart) {
            sb.append("-").append(rangeEnd + 1);
        }
        return sb.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @FXML
    private void onSelectAll() {
        if (!this.pdfService.isDocumentLoaded()) {
            return;
        }
        logger.info("User selected all pages");
        this.isBulkOperation = true;
        try {
            int pageCount = this.pdfService.getCurrentDocument().getPageCount();
            this.selectedPages.addAll(IntStream.range(0, pageCount).boxed().collect(Collectors.toSet()));
            for (PageThumbnailPanel panel : this.thumbnailPanels) {
                panel.setSelected(true);
            }
        }
        finally {
            this.isBulkOperation = false;
        }
        this.updateSelectedPagesList();
    }

    @FXML
    private void onDeselectAll() {
        logger.info("User deselected all pages");
        this.isBulkOperation = true;
        try {
            this.selectedPages.clear();
            for (PageThumbnailPanel panel : this.thumbnailPanels) {
                panel.setSelected(false);
            }
        }
        finally {
            this.isBulkOperation = false;
        }
        this.updateSelectedPagesList();
    }

    @FXML
    private void onRemoveSelected() {
        logger.warn("Remove selected not yet implemented");
        this.showWarning("This feature will be implemented soon");
    }

    @FXML
    private void onClearAll() {
        logger.info("User cleared all selected pages");
        this.selectedPages.clear();
        this.updateSelectedPagesList();
    }

    @FXML
    private void onBrowseOutput() {
        logger.info("User clicked browse for output file");
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Save Extracted PDF As");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        fileChooser.setInitialFileName("extracted.pdf");
        Stage stage = (Stage)this.browseButton.getScene().getWindow();
        File selectedFile = fileChooser.showSaveDialog(stage);
        if (selectedFile != null) {
            this.outputFileField.setText(selectedFile.getAbsolutePath());
            logger.debug("Output file selected: {}", (Object)selectedFile.getAbsolutePath());
        }
    }

    @FXML
    private void onExport() {
        logger.info("User clicked export");
        if (this.selectedPages.isEmpty()) {
            logger.warn("No pages selected for export");
            this.showWarning("Please select pages to export");
            return;
        }
        String outputPath = this.outputFileField.getText().trim();
        if (outputPath.isEmpty()) {
            logger.warn("No output file specified");
            this.showWarning("Please specify an output file");
            return;
        }
        final File outputFile = new File(outputPath);
        logger.info("Starting export to: {} with {} pages", (Object)outputFile.getAbsolutePath(), (Object)this.selectedPages.size());
        final HashMap<Integer, Integer> selectedRotations = new HashMap<Integer, Integer>();
        for (int pageIndex : this.selectedPages) {
            int rotation = this.pageRotations.getOrDefault(pageIndex, 0);
            if (rotation == 0) continue;
            selectedRotations.put(pageIndex, rotation);
        }
        Task<Void> exportTask = new Task<Void>(){

            @Override
            protected Void call() throws Exception {
                logger.debug("Background task: extracting pages with {} rotations", (Object)selectedRotations.size());
                PdfExtractorController.this.pdfService.extractPages(new HashSet<Integer>(PdfExtractorController.this.selectedPages), selectedRotations, outputFile);
                return null;
            }
        };
        exportTask.setOnSucceeded(event -> {
            logger.info("Export completed successfully");
            this.showInfo("PDF exported successfully");
        });
        exportTask.setOnFailed(event -> {
            logger.error("Export failed", exportTask.getException());
            this.showError("Export failed: " + exportTask.getException().getMessage());
        });
        new Thread(exportTask).start();
    }

    @FXML
    private void onRotateLeft() {
        this.rotateSelectedPages(-90);
    }

    @FXML
    private void onRotateRight() {
        this.rotateSelectedPages(90);
    }

    private void rotateSelectedPages(int degrees) {
        if (!this.pdfService.isDocumentLoaded()) {
            return;
        }
        HashSet<Integer> pagesToRotate = new HashSet<Integer>(this.selectedPages);
        if (pagesToRotate.isEmpty() && this.currentPreviewPage >= 0) {
            pagesToRotate.add(this.currentPreviewPage);
        }
        if (pagesToRotate.isEmpty()) {
            this.showWarning("Please select pages to rotate");
            return;
        }
        logger.info("Rotating {} pages by {} degrees", (Object)pagesToRotate.size(), (Object)degrees);
        Iterator iterator = pagesToRotate.iterator();
        while (iterator.hasNext()) {
            int pageIndex = (Integer)iterator.next();
            int currentRotation = this.pageRotations.getOrDefault(pageIndex, 0);
            int newRotation = (currentRotation + degrees + 360) % 360;
            this.pageRotations.put(pageIndex, newRotation);
            logger.debug("Page {} rotation: {} -> {}", pageIndex + 1, currentRotation, newRotation);
            this.updateThumbnailRotation(pageIndex);
        }
        if (pagesToRotate.contains(this.currentPreviewPage)) {
            this.updatePreview(this.currentPreviewPage);
        }
        this.showInfo("Rotated " + pagesToRotate.size() + " page(s)");
    }

    private void updateThumbnailRotation(int pageIndex) {
        if (pageIndex < 0 || pageIndex >= this.thumbnailPanels.size()) {
            return;
        }
        PageThumbnailPanel panel = this.thumbnailPanels.get(pageIndex);
        int rotation = this.pageRotations.getOrDefault(pageIndex, 0);
        panel.setRotation(rotation);
    }

    private void showError(String message) {
        logger.warn("Showing error notification: {}", (Object)message);
        this.showNotification(message, "#d32f2f");
    }

    private void showWarning(String message) {
        logger.warn("Showing warning notification: {}", (Object)message);
        this.showNotification(message, "#f57c00");
    }

    private void showInfo(String message) {
        logger.info("Showing info notification: {}", (Object)message);
        this.showNotification(message, "#388e3c");
    }

    private void showNotification(String message, String bgColor) {
        Platform.runLater(() -> {
            this.notificationLabel.setText(message);
            this.notificationLabel.setStyle("-fx-padding: 8; -fx-font-size: 12; -fx-text-fill: white; -fx-background-color: " + bgColor + ";");
            this.notificationLabel.setVisible(true);
            Timeline timeline = new Timeline(new KeyFrame(Duration.seconds(5.0), event -> this.notificationLabel.setVisible(false), new KeyValue[0]));
            timeline.play();
        });
    }

    private void loadBookmarks() {
        this.currentBookmarks = this.pdfService.getBookmarks();
        if (this.currentBookmarks.isEmpty()) {
            logger.debug("No bookmarks found in document");
            this.bookmarksTab.setDisable(true);
            return;
        }
        logger.info("Loaded {} bookmarks", (Object)this.currentBookmarks.size());
        this.bookmarksTab.setDisable(false);
        TreeItem root = new TreeItem();
        root.setExpanded(true);
        for (PdfBookmark bookmark : this.currentBookmarks) {
            TreeItem<PdfBookmark> item = this.createTreeItem(bookmark);
            root.getChildren().add(item);
        }
        this.bookmarkTreeView.setRoot(root);
        int maxLevel = this.getMaxBookmarkLevel(this.currentBookmarks, 0);
        SpinnerValueFactory.IntegerSpinnerValueFactory valueFactory = new SpinnerValueFactory.IntegerSpinnerValueFactory(1, maxLevel + 1, 1);
        this.levelSpinner.setValueFactory(valueFactory);
        this.updateBookmarkExportButton();
    }

    private int getMaxBookmarkLevel(List<PdfBookmark> bookmarks, int currentLevel) {
        int maxLevel = currentLevel;
        for (PdfBookmark bookmark : bookmarks) {
            if (!bookmark.hasChildren()) continue;
            int childMax = this.getMaxBookmarkLevel(bookmark.getChildren(), currentLevel + 1);
            maxLevel = Math.max(maxLevel, childMax);
        }
        return maxLevel;
    }

    private TreeItem<PdfBookmark> createTreeItem(PdfBookmark bookmark) {
        TreeItem<PdfBookmark> item = new TreeItem<PdfBookmark>(bookmark);
        item.setExpanded(true);
        for (PdfBookmark child : bookmark.getChildren()) {
            item.getChildren().add(this.createTreeItem(child));
        }
        return item;
    }

    private void updateBookmarkExportButton() {
        List<PdfBookmark> selected = this.getSelectedBookmarks();
        boolean hasSelection = !selected.isEmpty();
        boolean hasOutputDir = !this.outputDirField.getText().trim().isEmpty();
        this.exportBookmarksButton.setDisable(!hasSelection || !hasOutputDir);
        if (hasSelection) {
            this.exportBookmarksButton.setText("Export " + selected.size() + " Chapter(s)");
        } else {
            this.exportBookmarksButton.setText("Export by Chapters");
        }
    }

    private List<PdfBookmark> getSelectedBookmarks() {
        ArrayList<PdfBookmark> selected = new ArrayList<PdfBookmark>();
        this.collectSelectedBookmarks(this.currentBookmarks, selected);
        return selected;
    }

    private void collectSelectedBookmarks(List<PdfBookmark> bookmarks, List<PdfBookmark> selected) {
        for (PdfBookmark bookmark : bookmarks) {
            if (bookmark.isSelected()) {
                selected.add(bookmark);
            }
            this.collectSelectedBookmarks(bookmark.getChildren(), selected);
        }
    }

    @FXML
    private void onSelectAllBookmarks() {
        logger.info("Selecting all bookmarks");
        this.setAllBookmarksSelected(this.currentBookmarks, true);
        this.refreshBookmarkTree();
        this.updateBookmarkExportButton();
    }

    @FXML
    private void onDeselectAllBookmarks() {
        logger.info("Deselecting all bookmarks");
        this.setAllBookmarksSelected(this.currentBookmarks, false);
        this.refreshBookmarkTree();
        this.updateBookmarkExportButton();
    }

    @FXML
    private void onSelectLevel() {
        int targetLevel = this.levelSpinner.getValue();
        logger.info("Selecting bookmarks at level {}", (Object)targetLevel);
        this.setAllBookmarksSelected(this.currentBookmarks, false);
        this.selectBookmarksAtLevel(this.currentBookmarks, 0, targetLevel - 1);
        this.refreshBookmarkTree();
        this.updateBookmarkExportButton();
    }

    private void selectBookmarksAtLevel(List<PdfBookmark> bookmarks, int currentLevel, int targetLevel) {
        for (PdfBookmark bookmark : bookmarks) {
            if (currentLevel == targetLevel) {
                bookmark.setSelected(true);
            }
            if (!bookmark.hasChildren()) continue;
            this.selectBookmarksAtLevel(bookmark.getChildren(), currentLevel + 1, targetLevel);
        }
    }

    private void setAllBookmarksSelected(List<PdfBookmark> bookmarks, boolean selected) {
        for (PdfBookmark bookmark : bookmarks) {
            bookmark.setSelected(selected);
            this.setAllBookmarksSelected(bookmark.getChildren(), selected);
        }
    }

    private void refreshBookmarkTree() {
        if (this.bookmarkTreeView.getRoot() != null) {
            TreeItem root = new TreeItem();
            root.setExpanded(true);
            for (PdfBookmark bookmark : this.currentBookmarks) {
                TreeItem<PdfBookmark> item = this.createTreeItem(bookmark);
                root.getChildren().add(item);
            }
            this.bookmarkTreeView.setRoot(root);
        }
    }

    @FXML
    private void onBrowseOutputDir() {
        logger.info("User clicked browse for output directory");
        DirectoryChooser dirChooser = new DirectoryChooser();
        dirChooser.setTitle("Select Output Directory");
        Stage stage = (Stage)this.browseDirButton.getScene().getWindow();
        File selectedDir = dirChooser.showDialog(stage);
        if (selectedDir != null) {
            this.outputDirField.setText(selectedDir.getAbsolutePath());
            logger.debug("Output directory selected: {}", (Object)selectedDir.getAbsolutePath());
            this.updateBookmarkExportButton();
        }
    }

    @FXML
    private void onExportByBookmarks() {
        logger.info("User clicked export by bookmarks");
        final List<PdfBookmark> selected = this.getSelectedBookmarks();
        if (selected.isEmpty()) {
            logger.warn("No bookmarks selected for export");
            this.showWarning("Please select bookmarks to export");
            return;
        }
        String outputPath = this.outputDirField.getText().trim();
        if (outputPath.isEmpty()) {
            logger.warn("No output directory specified");
            this.showWarning("Please specify an output directory");
            return;
        }
        final File outputDir = new File(outputPath);
        logger.info("Starting export of {} bookmarks to: {}", (Object)selected.size(), (Object)outputDir.getAbsolutePath());
        Task<Void> exportTask = new Task<Void>(){

            @Override
            protected Void call() throws Exception {
                logger.debug("Background task: extracting by bookmarks");
                PdfExtractorController.this.pdfService.extractByBookmarks(selected, outputDir);
                return null;
            }
        };
        exportTask.setOnSucceeded(event -> {
            logger.info("Bookmark export completed successfully");
            this.showInfo("Exported " + selected.size() + " chapter(s) successfully");
        });
        exportTask.setOnFailed(event -> {
            logger.error("Bookmark export failed", exportTask.getException());
            this.showError("Export failed: " + exportTask.getException().getMessage());
        });
        new Thread(exportTask).start();
    }

    private static class CheckBoxTreeCell<T>
    extends TreeCell<T> {
        private CheckBoxTreeCell() {
        }
    }
}

