/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.ui.inserter;

import com.datmt.pdftools.model.JoinerSection;
import com.datmt.pdftools.service.PdfBulkInserterService;
import com.datmt.pdftools.util.CreditLinkHandler;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import javafx.application.Platform;
import javafx.fxml.FXML;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Alert;
import javafx.scene.control.Button;
import javafx.scene.control.CheckBox;
import javafx.scene.control.ComboBox;
import javafx.scene.control.Hyperlink;
import javafx.scene.control.Label;
import javafx.scene.control.ProgressBar;
import javafx.scene.control.RadioButton;
import javafx.scene.control.Spinner;
import javafx.scene.control.TextField;
import javafx.scene.control.ToggleGroup;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.stage.FileChooser;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfBulkInserterController {
    private static final Logger logger = LoggerFactory.getLogger(PdfBulkInserterController.class);
    private static final List<String> IMAGE_EXTENSIONS = Arrays.asList(".png", ".jpg", ".jpeg", ".gif", ".bmp");
    @FXML
    private Button selectSourceButton;
    @FXML
    private Label sourceFileLabel;
    @FXML
    private Button insertAllButton;
    @FXML
    private Button addFilesButton;
    @FXML
    private Button removeFileButton;
    @FXML
    private Button clearAllButton;
    @FXML
    private VBox filesContainer;
    @FXML
    private Label totalTargetsLabel;
    @FXML
    private ImageView sourcePreviewImage;
    @FXML
    private Label noSourceLabel;
    @FXML
    private Label sourceNameLabel;
    @FXML
    private Label sourceInfoLabel;
    @FXML
    private RadioButton atEndRadio;
    @FXML
    private RadioButton afterPageRadio;
    @FXML
    private RadioButton everyNPagesRadio;
    @FXML
    private ToggleGroup insertionModeGroup;
    @FXML
    private HBox afterPageOptions;
    @FXML
    private HBox everyNPagesOptions;
    @FXML
    private Spinner<Integer> afterPageSpinner;
    @FXML
    private Spinner<Integer> everyNPagesSpinner;
    @FXML
    private CheckBox insertAtEndIfShorterCheck;
    @FXML
    private VBox imageOptionsPane;
    @FXML
    private ComboBox<JoinerSection.PageSize> pageSizeCombo;
    @FXML
    private ComboBox<JoinerSection.FitOption> fitOptionCombo;
    @FXML
    private TextField suffixField;
    @FXML
    private VBox progressSection;
    @FXML
    private Label progressLabel;
    @FXML
    private ProgressBar progressBar;
    @FXML
    private Label progressDetailLabel;
    @FXML
    private VBox resultsSection;
    @FXML
    private VBox resultsContainer;
    @FXML
    private Label totalSuccessLabel;
    @FXML
    private Label totalFailedLabel;
    @FXML
    private Label totalInsertionsLabel;
    @FXML
    private Hyperlink creditLink;
    private File sourceFile;
    private boolean sourceIsPdf;
    private PDDocument sourceDocument;
    private final List<TargetFileItem> targetFileItems = new ArrayList<TargetFileItem>();
    private TargetFileItem selectedTargetItem;
    private final PdfBulkInserterService inserterService = new PdfBulkInserterService();
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    private boolean isInserting = false;

    @FXML
    public void initialize() {
        logger.info("Initializing PDF Bulk Inserter");
        this.setupWindowCloseHandler();
        this.setupListeners();
        this.setupComboBoxes();
        CreditLinkHandler.setup(this.creditLink);
        this.updateUI();
    }

    private void setupWindowCloseHandler() {
        Platform.runLater(() -> {
            if (this.selectSourceButton.getScene() != null && this.selectSourceButton.getScene().getWindow() != null) {
                this.selectSourceButton.getScene().getWindow().setOnCloseRequest(event -> this.cleanup());
            }
        });
    }

    private void setupListeners() {
        this.insertionModeGroup.selectedToggleProperty().addListener((obs, oldVal, newVal) -> this.updateModeOptionsVisibility());
    }

    private void setupComboBoxes() {
        this.pageSizeCombo.getItems().addAll((JoinerSection.PageSize[])JoinerSection.PageSize.values());
        this.pageSizeCombo.setValue(JoinerSection.PageSize.A4);
        this.fitOptionCombo.getItems().addAll((JoinerSection.FitOption[])JoinerSection.FitOption.values());
        this.fitOptionCombo.setValue(JoinerSection.FitOption.FIT_TO_PAGE);
    }

    private void updateModeOptionsVisibility() {
        this.afterPageOptions.setDisable(!this.afterPageRadio.isSelected());
        this.everyNPagesOptions.setDisable(!this.everyNPagesRadio.isSelected());
    }

    @FXML
    private void onSelectSource() {
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select PDF or Image to Insert");
        fileChooser.getExtensionFilters().addAll((FileChooser.ExtensionFilter[])new FileChooser.ExtensionFilter[]{new FileChooser.ExtensionFilter("PDF and Image Files", "*.pdf", "*.png", "*.jpg", "*.jpeg", "*.gif", "*.bmp"), new FileChooser.ExtensionFilter("PDF Files", "*.pdf"), new FileChooser.ExtensionFilter("Image Files", "*.png", "*.jpg", "*.jpeg", "*.gif", "*.bmp")});
        File file = fileChooser.showOpenDialog(this.selectSourceButton.getScene().getWindow());
        if (file != null) {
            this.loadSource(file);
        }
    }

    private void loadSource(File file) {
        this.closeSourceDocument();
        this.sourceFile = file;
        String fileName = file.getName().toLowerCase();
        this.sourceIsPdf = fileName.endsWith(".pdf");
        this.imageOptionsPane.setVisible(!this.sourceIsPdf);
        this.imageOptionsPane.setManaged(!this.sourceIsPdf);
        this.executor.submit(() -> {
            try {
                Image thumbnail;
                int pageCount;
                if (this.sourceIsPdf) {
                    this.sourceDocument = this.inserterService.loadSourcePdf(file);
                    pageCount = this.sourceDocument.getNumberOfPages();
                    thumbnail = this.inserterService.renderThumbnailFromDocument(this.sourceDocument, 0);
                } else {
                    pageCount = 1;
                    thumbnail = this.inserterService.renderThumbnail(file, false, 0);
                }
                int finalPageCount = pageCount;
                Image finalThumbnail = thumbnail;
                Platform.runLater(() -> {
                    this.sourcePreviewImage.setImage(finalThumbnail);
                    this.noSourceLabel.setVisible(false);
                    this.sourceNameLabel.setText(file.getName());
                    this.sourceInfoLabel.setText((String)(this.sourceIsPdf ? finalPageCount + " page(s)" : "1 image"));
                    this.sourceFileLabel.setText(file.getName());
                    this.updateUI();
                });
            }
            catch (IOException e) {
                logger.error("Error loading source: {}", (Object)e.getMessage());
                Platform.runLater(() -> {
                    this.showError("Failed to load source", e.getMessage());
                    this.clearSource();
                });
            }
        });
    }

    private void closeSourceDocument() {
        if (this.sourceDocument != null) {
            try {
                this.sourceDocument.close();
            }
            catch (IOException e) {
                logger.error("Error closing source document: {}", (Object)e.getMessage());
            }
            this.sourceDocument = null;
        }
    }

    private void clearSource() {
        this.closeSourceDocument();
        this.sourceFile = null;
        this.sourcePreviewImage.setImage(null);
        this.noSourceLabel.setVisible(true);
        this.sourceNameLabel.setText("-");
        this.sourceInfoLabel.setText("-");
        this.sourceFileLabel.setText("No source selected");
        this.imageOptionsPane.setVisible(false);
        this.imageOptionsPane.setManaged(false);
        this.updateUI();
    }

    @FXML
    private void onAddFiles() {
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select Target PDF Files");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        List<File> files = fileChooser.showOpenMultipleDialog(this.addFilesButton.getScene().getWindow());
        if (files != null) {
            for (File file : files) {
                this.addTargetFile(file);
            }
            this.updateUI();
        }
    }

    private void addTargetFile(File file) {
        for (TargetFileItem item : this.targetFileItems) {
            if (!item.getFile().getAbsolutePath().equals(file.getAbsolutePath())) continue;
            logger.debug("File already added: {}", (Object)file.getName());
            return;
        }
        if (this.sourceFile != null && this.sourceFile.getAbsolutePath().equals(file.getAbsolutePath())) {
            logger.debug("Cannot add source file as target: {}", (Object)file.getName());
            return;
        }
        this.executor.submit(() -> {
            try {
                int pageCount = this.inserterService.getPageCount(file);
                Platform.runLater(() -> {
                    TargetFileItem item = new TargetFileItem(file, pageCount, this::onTargetSelected, this::onTargetRemove);
                    this.targetFileItems.add(item);
                    this.filesContainer.getChildren().add(item.getNode());
                    this.updateUI();
                    logger.info("Added target file: {} ({} pages)", (Object)file.getName(), (Object)pageCount);
                });
            }
            catch (IOException e) {
                logger.error("Error loading target file: {}", (Object)e.getMessage());
                Platform.runLater(() -> this.showError("Failed to load", file.getName() + ": " + e.getMessage()));
            }
        });
    }

    private void onTargetSelected(TargetFileItem item) {
        if (this.selectedTargetItem != null) {
            this.selectedTargetItem.setSelected(false);
        }
        this.selectedTargetItem = item;
        item.setSelected(true);
        this.updateUI();
    }

    private void onTargetRemove(TargetFileItem item) {
        this.targetFileItems.remove(item);
        this.filesContainer.getChildren().remove(item.getNode());
        if (this.selectedTargetItem == item) {
            this.selectedTargetItem = null;
        }
        this.updateUI();
    }

    @FXML
    private void onRemoveFile() {
        if (this.selectedTargetItem != null) {
            this.onTargetRemove(this.selectedTargetItem);
        }
    }

    @FXML
    private void onClearAll() {
        this.targetFileItems.clear();
        this.filesContainer.getChildren().clear();
        this.selectedTargetItem = null;
        this.resultsSection.setVisible(false);
        this.resultsSection.setManaged(false);
        this.resultsContainer.getChildren().clear();
        this.updateUI();
    }

    @FXML
    private void onInsertAll() {
        if (this.sourceFile == null || this.targetFileItems.isEmpty() || this.isInserting) {
            return;
        }
        PdfBulkInserterService.InsertionOptions options = this.buildOptions();
        this.isInserting = true;
        this.updateUI();
        this.progressSection.setVisible(true);
        this.progressSection.setManaged(true);
        this.progressBar.setProgress(0.0);
        this.resultsSection.setVisible(false);
        this.resultsSection.setManaged(false);
        this.resultsContainer.getChildren().clear();
        ArrayList<File> targetFiles = new ArrayList<File>();
        for (TargetFileItem item : this.targetFileItems) {
            targetFiles.add(item.getFile());
        }
        int totalFiles = targetFiles.size();
        this.executor.submit(() -> {
            ArrayList<PdfBulkInserterService.InsertResult> allResults = new ArrayList<PdfBulkInserterService.InsertResult>();
            PDDocument sourceDoc = null;
            try {
                sourceDoc = this.sourceIsPdf ? this.inserterService.loadSourcePdf(this.sourceFile) : this.inserterService.createPdfFromImage(this.sourceFile, options);
                PDDocument finalSourceDoc = sourceDoc;
                int i = 0;
                while (i < totalFiles) {
                    File targetFile = (File)targetFiles.get(i);
                    int fileIndex = i++;
                    Platform.runLater(() -> {
                        this.progressLabel.setText("Processing: " + targetFile.getName() + " (" + (fileIndex + 1) + "/" + totalFiles + ")");
                        this.progressBar.setProgress((double)fileIndex / (double)totalFiles);
                    });
                    PdfBulkInserterService.InsertResult result = this.inserterService.insertIntoTarget(targetFile, finalSourceDoc, options);
                    allResults.add(result);
                    PdfBulkInserterService.InsertResult finalResult = result;
                    Platform.runLater(() -> {
                        if (finalResult.isSuccess()) {
                            this.progressDetailLabel.setText(String.format("Inserted %d time(s), %d -> %d pages", finalResult.getInsertionsPerformed(), finalResult.getOriginalPageCount(), finalResult.getNewPageCount()));
                        } else {
                            this.progressDetailLabel.setText("Error: " + finalResult.getErrorMessage());
                        }
                    });
                }
            }
            catch (IOException e) {
                logger.error("Error during insertion: {}", (Object)e.getMessage());
                Platform.runLater(() -> this.showError("Insertion Error", e.getMessage()));
            }
            finally {
                if (sourceDoc != null && sourceDoc != this.sourceDocument) {
                    try {
                        sourceDoc.close();
                    }
                    catch (IOException e) {
                        logger.error("Error closing source: {}", (Object)e.getMessage());
                    }
                }
            }
            Platform.runLater(() -> {
                this.showResults(allResults);
                this.isInserting = false;
                this.progressSection.setVisible(false);
                this.progressSection.setManaged(false);
                this.updateUI();
            });
        });
    }

    private PdfBulkInserterService.InsertionOptions buildOptions() {
        PdfBulkInserterService.InsertionOptions options = new PdfBulkInserterService.InsertionOptions();
        if (this.atEndRadio.isSelected()) {
            options.setMode(PdfBulkInserterService.InsertionMode.AT_END);
        } else if (this.afterPageRadio.isSelected()) {
            options.setMode(PdfBulkInserterService.InsertionMode.AFTER_PAGE);
            options.setPageNumber(this.afterPageSpinner.getValue());
        } else if (this.everyNPagesRadio.isSelected()) {
            options.setMode(PdfBulkInserterService.InsertionMode.EVERY_N_PAGES);
            options.setInterval(this.everyNPagesSpinner.getValue());
        }
        options.setInsertAtEndIfShorter(this.insertAtEndIfShorterCheck.isSelected());
        options.setOutputSuffix(this.suffixField.getText());
        options.setPageSize((JoinerSection.PageSize)((Object)this.pageSizeCombo.getValue()));
        options.setFitOption((JoinerSection.FitOption)((Object)this.fitOptionCombo.getValue()));
        return options;
    }

    private void showResults(List<PdfBulkInserterService.InsertResult> results) {
        this.resultsContainer.getChildren().clear();
        int successCount = 0;
        int failedCount = 0;
        int totalInsertions = 0;
        for (PdfBulkInserterService.InsertResult result : results) {
            HBox row = new HBox(10.0);
            row.setAlignment(Pos.CENTER_LEFT);
            Label nameLabel = new Label(result.getInputFile().getName());
            nameLabel.setMaxWidth(200.0);
            nameLabel.setStyle("-fx-font-size: 11;");
            if (result.isSuccess()) {
                ++successCount;
                totalInsertions += result.getInsertionsPerformed();
                String statusText = String.format("%d -> %d pages (+%d)", result.getOriginalPageCount(), result.getNewPageCount(), result.getInsertionsPerformed());
                Label statusLabel = new Label(statusText);
                statusLabel.setStyle("-fx-font-size: 11; -fx-text-fill: #4CAF50;");
                row.getChildren().addAll((Node[])new Node[]{nameLabel, statusLabel});
            } else {
                ++failedCount;
                Label errorLabel = new Label("Failed: " + result.getErrorMessage());
                errorLabel.setStyle("-fx-font-size: 11; -fx-text-fill: #cc0000;");
                row.getChildren().addAll((Node[])new Node[]{nameLabel, errorLabel});
            }
            this.resultsContainer.getChildren().add(row);
        }
        this.totalSuccessLabel.setText("Success: " + successCount);
        this.totalFailedLabel.setText("Failed: " + failedCount);
        this.totalFailedLabel.setStyle(failedCount > 0 ? "-fx-font-size: 13; -fx-text-fill: #cc0000;" : "-fx-font-size: 13;");
        this.totalInsertionsLabel.setText("Total insertions: " + totalInsertions);
        this.resultsSection.setVisible(true);
        this.resultsSection.setManaged(true);
        logger.info("Insertion complete: {}/{} files successful, {} total insertions", successCount, results.size(), totalInsertions);
    }

    private void updateUI() {
        boolean hasSource = this.sourceFile != null;
        boolean hasTargets = !this.targetFileItems.isEmpty();
        boolean hasSelection = this.selectedTargetItem != null;
        this.insertAllButton.setDisable(!hasSource || !hasTargets || this.isInserting);
        this.removeFileButton.setDisable(!hasSelection || this.isInserting);
        this.clearAllButton.setDisable(!hasTargets || this.isInserting);
        this.addFilesButton.setDisable(this.isInserting);
        this.selectSourceButton.setDisable(this.isInserting);
        int totalPages = 0;
        for (TargetFileItem item : this.targetFileItems) {
            totalPages += item.getPageCount();
        }
        this.totalTargetsLabel.setText(String.format("Total: %d files, %d pages", this.targetFileItems.size(), totalPages));
    }

    private void showError(String title, String message) {
        Alert alert = new Alert(Alert.AlertType.ERROR);
        alert.setTitle(title);
        alert.setHeaderText(null);
        alert.setContentText(message);
        alert.showAndWait();
    }

    public void cleanup() {
        logger.info("Cleaning up PDF Bulk Inserter");
        this.executor.shutdownNow();
        this.closeSourceDocument();
        this.targetFileItems.clear();
        this.filesContainer.getChildren().clear();
    }

    private static class TargetFileItem {
        private final File file;
        private final int pageCount;
        private final HBox node;
        private boolean selected = false;

        public TargetFileItem(File file, int pageCount, Consumer<TargetFileItem> onSelect, Consumer<TargetFileItem> onRemove) {
            this.file = file;
            this.pageCount = pageCount;
            this.node = this.createNode(onSelect, onRemove);
        }

        private HBox createNode(Consumer<TargetFileItem> onSelect, Consumer<TargetFileItem> onRemove) {
            HBox box = new HBox(10.0);
            box.setAlignment(Pos.CENTER_LEFT);
            box.setStyle("-fx-padding: 8; -fx-background-color: white; -fx-border-color: #e0e0e0; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            Label nameLabel = new Label(this.file.getName());
            nameLabel.setStyle("-fx-font-size: 13;");
            nameLabel.setMaxWidth(250.0);
            HBox.setHgrow(nameLabel, Priority.ALWAYS);
            Label pagesLabel = new Label(this.pageCount + " pages");
            pagesLabel.setStyle("-fx-font-size: 12; -fx-text-fill: #666666;");
            Button removeBtn = new Button("X");
            removeBtn.setStyle("-fx-font-size: 10; -fx-padding: 2 6; -fx-background-color: #ff5252; -fx-text-fill: white; -fx-cursor: hand;");
            removeBtn.setOnAction(e -> onRemove.accept(this));
            box.getChildren().addAll((Node[])new Node[]{nameLabel, pagesLabel, removeBtn});
            box.setOnMouseClicked(e -> onSelect.accept(this));
            return box;
        }

        public File getFile() {
            return this.file;
        }

        public int getPageCount() {
            return this.pageCount;
        }

        public HBox getNode() {
            return this.node;
        }

        public void setSelected(boolean selected) {
            this.selected = selected;
            if (selected) {
                this.node.setStyle("-fx-padding: 8; -fx-background-color: #e3f2fd; -fx-border-color: #2196F3; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            } else {
                this.node.setStyle("-fx-padding: 8; -fx-background-color: white; -fx-border-color: #e0e0e0; -fx-border-radius: 4; -fx-background-radius: 4; -fx-cursor: hand;");
            }
        }
    }
}

