/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.ui.joiner;

import com.datmt.pdftools.model.JoinerFile;
import com.datmt.pdftools.model.JoinerSection;
import com.datmt.pdftools.service.PdfMergeService;
import com.datmt.pdftools.ui.joiner.components.FileListItem;
import com.datmt.pdftools.ui.joiner.components.SectionListItem;
import com.datmt.pdftools.util.CreditLinkHandler;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javafx.animation.KeyFrame;
import javafx.animation.KeyValue;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.concurrent.Task;
import javafx.fxml.FXML;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.ComboBox;
import javafx.scene.control.Hyperlink;
import javafx.scene.control.Label;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.TextField;
import javafx.scene.control.ToggleButton;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.stage.FileChooser;
import javafx.stage.Stage;
import javafx.util.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfJoinerController {
    private static final Logger logger = LoggerFactory.getLogger(PdfJoinerController.class);
    private static final int THUMBNAIL_SIZE = 80;
    @FXML
    private Button addFilesButton;
    @FXML
    private Label fileCountLabel;
    @FXML
    private Button joinButton;
    @FXML
    private Label notificationLabel;
    @FXML
    private ScrollPane fileListScrollPane;
    @FXML
    private VBox fileListContainer;
    @FXML
    private TextField pageRangeField;
    @FXML
    private Button addSectionButton;
    @FXML
    private ScrollPane sectionListScrollPane;
    @FXML
    private VBox sectionListContainer;
    @FXML
    private Label totalPagesLabel;
    @FXML
    private Button clearSectionsButton;
    @FXML
    private ToggleButton gridViewToggle;
    @FXML
    private ToggleButton singleViewToggle;
    @FXML
    private ScrollPane gridPreviewScrollPane;
    @FXML
    private FlowPane gridPreviewContainer;
    @FXML
    private VBox singlePreviewContainer;
    @FXML
    private ScrollPane singlePreviewScrollPane;
    @FXML
    private StackPane singlePreviewPane;
    @FXML
    private ImageView singlePreviewImage;
    @FXML
    private Button prevPageButton;
    @FXML
    private Button nextPageButton;
    @FXML
    private Label currentPageLabel;
    @FXML
    private VBox imageOptionsPane;
    @FXML
    private ComboBox<JoinerSection.PageSize> pageSizeCombo;
    @FXML
    private ComboBox<JoinerSection.FitOption> fitOptionCombo;
    @FXML
    private ComboBox<JoinerSection.Rotation> rotationCombo;
    @FXML
    private TextField outputFileField;
    @FXML
    private Button browseButton;
    @FXML
    private Hyperlink creditLink;
    private final PdfMergeService mergeService = new PdfMergeService();
    private final List<JoinerFile> loadedFiles = new ArrayList<JoinerFile>();
    private final List<JoinerSection> sections = new ArrayList<JoinerSection>();
    private final List<FileListItem> fileListItems = new ArrayList<FileListItem>();
    private final List<SectionListItem> sectionListItems = new ArrayList<SectionListItem>();
    private final ExecutorService renderExecutor = Executors.newFixedThreadPool(4);
    private FileListItem selectedFileItem;
    private SectionListItem selectedSectionItem;
    private int currentPreviewPage = 0;
    private boolean isGridView = true;

    @FXML
    public void initialize() {
        logger.trace("Initializing PdfJoinerController");
        this.setupImageOptionsComboBoxes();
        this.setupViewToggle();
        this.setupWindowCloseHandler();
        CreditLinkHandler.setup(this.creditLink);
        this.updateUI();
        logger.debug("Controller initialization complete");
    }

    private void setupWindowCloseHandler() {
        Platform.runLater(() -> {
            if (this.addFilesButton.getScene() != null && this.addFilesButton.getScene().getWindow() != null) {
                this.addFilesButton.getScene().getWindow().setOnCloseRequest(event -> this.cleanup());
            }
        });
    }

    public void cleanup() {
        logger.info("Cleaning up PdfJoinerController resources");
        this.renderExecutor.shutdownNow();
        for (JoinerFile file : this.loadedFiles) {
            try {
                file.close();
            }
            catch (Exception e) {
                logger.error("Error closing file: {}", (Object)file.getFileName(), (Object)e);
            }
        }
        this.loadedFiles.clear();
        this.sections.clear();
        this.fileListItems.clear();
        this.sectionListItems.clear();
        this.fileListContainer.getChildren().clear();
        this.sectionListContainer.getChildren().clear();
        this.gridPreviewContainer.getChildren().clear();
        logger.info("Cleanup complete");
    }

    private void setupImageOptionsComboBoxes() {
        this.pageSizeCombo.getItems().addAll((JoinerSection.PageSize[])JoinerSection.PageSize.values());
        this.pageSizeCombo.setValue(JoinerSection.PageSize.A4);
        this.pageSizeCombo.setOnAction(e -> this.onImageOptionChanged());
        this.fitOptionCombo.getItems().addAll((JoinerSection.FitOption[])JoinerSection.FitOption.values());
        this.fitOptionCombo.setValue(JoinerSection.FitOption.FIT_TO_PAGE);
        this.fitOptionCombo.setOnAction(e -> this.onImageOptionChanged());
        this.rotationCombo.getItems().addAll((JoinerSection.Rotation[])JoinerSection.Rotation.values());
        this.rotationCombo.setValue(JoinerSection.Rotation.NONE);
        this.rotationCombo.setOnAction(e -> this.onImageOptionChanged());
    }

    private void setupViewToggle() {
        this.gridViewToggle.setSelected(true);
        this.singleViewToggle.setSelected(false);
    }

    private void updateUI() {
        boolean hasFiles = !this.loadedFiles.isEmpty();
        boolean hasSections = !this.sections.isEmpty();
        boolean hasSelectedFile = this.selectedFileItem != null;
        this.fileCountLabel.setText((String)(hasFiles ? this.loadedFiles.size() + " file(s)" : "No files loaded"));
        this.addSectionButton.setDisable(!hasSelectedFile);
        this.clearSectionsButton.setDisable(!hasSections);
        this.joinButton.setDisable(!hasSections || this.outputFileField.getText().trim().isEmpty());
        this.updateTotalPagesLabel();
    }

    private void updateTotalPagesLabel() {
        int totalPages = this.sections.stream().mapToInt(JoinerSection::getPageCount).sum();
        this.totalPagesLabel.setText(totalPages + " pages");
    }

    @FXML
    private void onAddFiles() {
        logger.info("User clicked Add Files");
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select PDF or Image Files");
        fileChooser.getExtensionFilters().addAll((FileChooser.ExtensionFilter[])new FileChooser.ExtensionFilter[]{new FileChooser.ExtensionFilter("All Supported", "*.pdf", "*.png", "*.jpg", "*.jpeg", "*.gif", "*.bmp"), new FileChooser.ExtensionFilter("PDF Files", "*.pdf"), new FileChooser.ExtensionFilter("Image Files", "*.png", "*.jpg", "*.jpeg", "*.gif", "*.bmp")});
        Stage stage = (Stage)this.addFilesButton.getScene().getWindow();
        List<File> selectedFiles = fileChooser.showOpenMultipleDialog(stage);
        if (selectedFiles != null && !selectedFiles.isEmpty()) {
            this.loadFiles(selectedFiles);
        }
    }

    private void loadFiles(List<File> files) {
        logger.info("Loading {} files", (Object)files.size());
        for (final File file : files) {
            Task<JoinerFile> loadTask = new Task<JoinerFile>(){

                @Override
                protected JoinerFile call() throws Exception {
                    String name = file.getName().toLowerCase();
                    if (name.endsWith(".pdf")) {
                        return PdfJoinerController.this.mergeService.loadPdfFile(file);
                    }
                    return PdfJoinerController.this.mergeService.loadImageFile(file);
                }
            };
            loadTask.setOnSucceeded(event -> {
                JoinerFile joinerFile = (JoinerFile)loadTask.getValue();
                this.loadedFiles.add(joinerFile);
                FileListItem item = new FileListItem(joinerFile, this::onFileSelected, this::onFileRemoved);
                this.fileListItems.add(item);
                this.fileListContainer.getChildren().add(item);
                this.loadFileThumbnail(item, joinerFile);
                this.updateUI();
                this.showInfo("Loaded: " + joinerFile.getFileName());
            });
            loadTask.setOnFailed(event -> {
                logger.error("Failed to load file: {}", (Object)file.getName(), (Object)loadTask.getException());
                this.showError("Failed to load: " + file.getName());
            });
            new Thread(loadTask).start();
        }
    }

    private void loadFileThumbnail(FileListItem item, JoinerFile joinerFile) {
        CompletableFuture.runAsync(() -> {
            try {
                Image thumbnail = joinerFile.isPdf() ? this.mergeService.renderPdfThumbnail(joinerFile, 0) : joinerFile.getImage();
                Platform.runLater(() -> item.setThumbnail(thumbnail));
            }
            catch (Exception e) {
                logger.error("Failed to load thumbnail for: {}", (Object)joinerFile.getFileName(), (Object)e);
            }
        }, this.renderExecutor);
    }

    private void onFileSelected(FileListItem item) {
        logger.debug("File selected: {}", (Object)item.getJoinerFile().getFileName());
        if (this.selectedFileItem != null) {
            this.selectedFileItem.setFileSelected(false);
        }
        this.selectedFileItem = item;
        item.setFileSelected(true);
        this.pageRangeField.setText("all");
        boolean isImage = item.getJoinerFile().isImage();
        this.imageOptionsPane.setVisible(isImage);
        this.imageOptionsPane.setManaged(isImage);
        this.updateUI();
    }

    private void onFileRemoved(FileListItem item) {
        logger.info("Removing file: {}", (Object)item.getJoinerFile().getFileName());
        JoinerFile joinerFile = item.getJoinerFile();
        this.sections.removeIf(s -> s.getSourceFile() == joinerFile);
        this.rebuildSectionList();
        this.loadedFiles.remove(joinerFile);
        this.fileListItems.remove(item);
        this.fileListContainer.getChildren().remove(item);
        try {
            joinerFile.close();
        }
        catch (Exception e) {
            logger.error("Error closing file: {}", (Object)joinerFile.getFileName(), (Object)e);
        }
        if (this.selectedFileItem == item) {
            this.selectedFileItem = null;
        }
        this.updateUI();
    }

    @FXML
    private void onAddSection() {
        if (this.selectedFileItem == null) {
            this.showWarning("Please select a file first");
            return;
        }
        JoinerFile joinerFile = this.selectedFileItem.getJoinerFile();
        String rangeInput = this.pageRangeField.getText().trim();
        logger.info("Adding section from {} with range: {}", (Object)joinerFile.getFileName(), (Object)rangeInput);
        try {
            JoinerSection section = this.createSection(joinerFile, rangeInput);
            if (joinerFile.isImage()) {
                section.setPageSize((JoinerSection.PageSize)((Object)this.pageSizeCombo.getValue()));
                section.setFitOption((JoinerSection.FitOption)((Object)this.fitOptionCombo.getValue()));
                section.setRotation((JoinerSection.Rotation)((Object)this.rotationCombo.getValue()));
            }
            this.sections.add(section);
            this.addSectionToList(section, this.sections.size());
            this.updateUI();
            this.showInfo("Added section: " + section.toString());
        }
        catch (IllegalArgumentException e) {
            this.showWarning(e.getMessage());
        }
    }

    private JoinerSection createSection(JoinerFile joinerFile, String rangeInput) {
        if (rangeInput.isEmpty() || rangeInput.equalsIgnoreCase("all")) {
            return new JoinerSection(joinerFile);
        }
        if (rangeInput.contains("-")) {
            String[] parts = rangeInput.split("-");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Invalid range format: " + rangeInput);
            }
            int start = Integer.parseInt(parts[0].trim()) - 1;
            int end = Integer.parseInt(parts[1].trim()) - 1;
            if (start < 0 || end >= joinerFile.getPageCount() || start > end) {
                throw new IllegalArgumentException("Invalid page range: " + rangeInput);
            }
            return new JoinerSection(joinerFile, start, end);
        }
        int page = Integer.parseInt(rangeInput.trim()) - 1;
        if (page < 0 || page >= joinerFile.getPageCount()) {
            throw new IllegalArgumentException("Invalid page number: " + rangeInput);
        }
        return new JoinerSection(joinerFile, page, page);
    }

    private void addSectionToList(JoinerSection section, int index) {
        SectionListItem item = new SectionListItem(section, index, this::onSectionSelected, this::onSectionMoveUp, this::onSectionMoveDown, this::onSectionRotateLeft, this::onSectionRotateRight, this::onSectionRemoved);
        this.sectionListItems.add(item);
        this.sectionListContainer.getChildren().add(item);
    }

    private void rebuildSectionList() {
        this.sectionListContainer.getChildren().clear();
        this.sectionListItems.clear();
        this.selectedSectionItem = null;
        for (int i = 0; i < this.sections.size(); ++i) {
            this.addSectionToList(this.sections.get(i), i + 1);
        }
        this.updateUI();
    }

    private void onSectionSelected(SectionListItem item) {
        logger.debug("Section selected: {}", (Object)item.getSection().toString());
        if (this.selectedSectionItem != null) {
            this.selectedSectionItem.setSectionSelected(false);
        }
        this.selectedSectionItem = item;
        item.setSectionSelected(true);
        boolean isImage = item.getSection().getSourceFile().isImage();
        this.imageOptionsPane.setVisible(isImage);
        this.imageOptionsPane.setManaged(isImage);
        if (isImage) {
            JoinerSection section = item.getSection();
            this.pageSizeCombo.setValue(section.getPageSize());
            this.fitOptionCombo.setValue(section.getFitOption());
            this.rotationCombo.setValue(section.getRotation());
        }
        this.currentPreviewPage = 0;
        this.updatePreview();
    }

    private void onSectionMoveUp(SectionListItem item) {
        int index = this.sectionListItems.indexOf(item);
        if (index > 0) {
            logger.debug("Moving section up from index {}", (Object)index);
            JoinerSection section = this.sections.remove(index);
            this.sections.add(index - 1, section);
            this.rebuildSectionList();
        }
    }

    private void onSectionMoveDown(SectionListItem item) {
        int index = this.sectionListItems.indexOf(item);
        if (index < this.sections.size() - 1) {
            logger.debug("Moving section down from index {}", (Object)index);
            JoinerSection section = this.sections.remove(index);
            this.sections.add(index + 1, section);
            this.rebuildSectionList();
        }
    }

    private void onSectionRemoved(SectionListItem item) {
        logger.info("Removing section: {}", (Object)item.getSection().toString());
        int index = this.sectionListItems.indexOf(item);
        this.sections.remove(index);
        this.rebuildSectionList();
    }

    private void onSectionRotateLeft(SectionListItem item) {
        this.rotateSectionBy(item, -90);
    }

    private void onSectionRotateRight(SectionListItem item) {
        this.rotateSectionBy(item, 90);
    }

    private void rotateSectionBy(SectionListItem item, int degrees) {
        JoinerSection section = item.getSection();
        JoinerSection.Rotation currentRotation = section.getRotation();
        int currentDegrees = currentRotation.getDegrees();
        int newDegrees = (currentDegrees + degrees + 360) % 360;
        JoinerSection.Rotation newRotation = JoinerSection.Rotation.NONE;
        for (JoinerSection.Rotation r : JoinerSection.Rotation.values()) {
            if (r.getDegrees() != newDegrees) continue;
            newRotation = r;
            break;
        }
        section.setRotation(newRotation);
        logger.debug("Section rotation changed: {} -> {}", (Object)currentDegrees, (Object)newDegrees);
        int sectionIndex = this.sections.indexOf(section);
        this.rebuildSectionList();
        if (sectionIndex >= 0 && sectionIndex < this.sectionListItems.size()) {
            SectionListItem newItem = this.sectionListItems.get(sectionIndex);
            this.onSectionSelected(newItem);
        }
        this.showInfo("Rotated section to " + newDegrees + "\u00b0");
    }

    @FXML
    private void onClearSections() {
        logger.info("Clearing all sections");
        this.sections.clear();
        this.rebuildSectionList();
        this.gridPreviewContainer.getChildren().clear();
    }

    private void onImageOptionChanged() {
        if (this.selectedSectionItem != null && this.selectedSectionItem.getSection().getSourceFile().isImage()) {
            JoinerSection section = this.selectedSectionItem.getSection();
            section.setPageSize((JoinerSection.PageSize)((Object)this.pageSizeCombo.getValue()));
            section.setFitOption((JoinerSection.FitOption)((Object)this.fitOptionCombo.getValue()));
            section.setRotation((JoinerSection.Rotation)((Object)this.rotationCombo.getValue()));
            logger.debug("Updated image options for section: {}", (Object)section.toString());
        }
    }

    @FXML
    private void onToggleGridView() {
        this.isGridView = true;
        this.gridViewToggle.setSelected(true);
        this.singleViewToggle.setSelected(false);
        this.gridPreviewScrollPane.setVisible(true);
        this.gridPreviewScrollPane.setManaged(true);
        this.singlePreviewContainer.setVisible(false);
        this.singlePreviewContainer.setManaged(false);
        this.updatePreview();
    }

    @FXML
    private void onToggleSingleView() {
        this.isGridView = false;
        this.gridViewToggle.setSelected(false);
        this.singleViewToggle.setSelected(true);
        this.gridPreviewScrollPane.setVisible(false);
        this.gridPreviewScrollPane.setManaged(false);
        this.singlePreviewContainer.setVisible(true);
        this.singlePreviewContainer.setManaged(true);
        this.updatePreview();
    }

    private void updatePreview() {
        if (this.selectedSectionItem == null) {
            this.gridPreviewContainer.getChildren().clear();
            return;
        }
        JoinerSection section = this.selectedSectionItem.getSection();
        if (this.isGridView) {
            this.updateGridPreview(section);
        } else {
            this.updateSinglePreview(section);
        }
    }

    private void updateGridPreview(JoinerSection section) {
        this.gridPreviewContainer.getChildren().clear();
        JoinerFile sourceFile = section.getSourceFile();
        int rotation = section.getRotation().getDegrees();
        int i = section.getStartPage();
        while (i <= section.getEndPage()) {
            int pageIndex = i++;
            ImageView thumb = new ImageView();
            thumb.setPreserveRatio(true);
            thumb.setFitWidth(80.0);
            thumb.setFitHeight(80.0);
            thumb.setRotate(rotation);
            VBox container = new VBox(5.0);
            container.setStyle("-fx-padding: 5; -fx-border-color: #e0e0e0; -fx-border-radius: 3;");
            Label pageLabel = new Label("Page " + (pageIndex + 1));
            pageLabel.setStyle("-fx-font-size: 10;");
            container.getChildren().addAll((Node[])new Node[]{thumb, pageLabel});
            this.gridPreviewContainer.getChildren().add(container);
            CompletableFuture.runAsync(() -> {
                try {
                    Image image = sourceFile.isPdf() ? this.mergeService.renderPdfThumbnail(sourceFile, pageIndex) : sourceFile.getImage();
                    Platform.runLater(() -> thumb.setImage(image));
                }
                catch (Exception e) {
                    logger.error("Failed to load preview thumbnail", e);
                }
            }, this.renderExecutor);
        }
    }

    private void updateSinglePreview(JoinerSection section) {
        final JoinerFile sourceFile = section.getSourceFile();
        int pageCount = section.getPageCount();
        this.currentPreviewPage = Math.max(0, Math.min(this.currentPreviewPage, pageCount - 1));
        final int actualPage = section.getStartPage() + this.currentPreviewPage;
        this.currentPageLabel.setText("Page " + (this.currentPreviewPage + 1) + " of " + pageCount);
        this.prevPageButton.setDisable(this.currentPreviewPage == 0);
        this.nextPageButton.setDisable(this.currentPreviewPage >= pageCount - 1);
        Task<Image> previewTask = new Task<Image>(){

            @Override
            protected Image call() throws Exception {
                if (sourceFile.isPdf()) {
                    return PdfJoinerController.this.mergeService.renderPdfPreview(sourceFile, actualPage);
                }
                return sourceFile.getImage();
            }
        };
        previewTask.setOnSucceeded(e -> {
            Image image = (Image)previewTask.getValue();
            this.singlePreviewImage.setImage(image);
            this.singlePreviewImage.setFitWidth(this.singlePreviewPane.getWidth() - 20.0);
            this.singlePreviewImage.setRotate(section.getRotation().getDegrees());
        });
        previewTask.setOnFailed(e -> logger.error("Failed to render preview", previewTask.getException()));
        new Thread(previewTask).start();
    }

    @FXML
    private void onPreviousPage() {
        if (this.currentPreviewPage > 0) {
            --this.currentPreviewPage;
            this.updatePreview();
        }
    }

    @FXML
    private void onNextPage() {
        int pageCount;
        if (this.selectedSectionItem != null && this.currentPreviewPage < (pageCount = this.selectedSectionItem.getSection().getPageCount()) - 1) {
            ++this.currentPreviewPage;
            this.updatePreview();
        }
    }

    @FXML
    private void onBrowseOutput() {
        logger.info("User clicked browse for output file");
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Save Joined PDF As");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        fileChooser.setInitialFileName("joined.pdf");
        Stage stage = (Stage)this.browseButton.getScene().getWindow();
        File selectedFile = fileChooser.showSaveDialog(stage);
        if (selectedFile != null) {
            this.outputFileField.setText(selectedFile.getAbsolutePath());
            this.updateUI();
        }
    }

    @FXML
    private void onJoin() {
        if (this.sections.isEmpty()) {
            this.showWarning("Please add sections to join");
            return;
        }
        String outputPath = this.outputFileField.getText().trim();
        if (outputPath.isEmpty()) {
            this.showWarning("Please specify an output file");
            return;
        }
        final File outputFile = new File(outputPath);
        logger.info("Starting join operation with {} sections to: {}", (Object)this.sections.size(), (Object)outputFile.getAbsolutePath());
        this.joinButton.setDisable(true);
        Task<Void> joinTask = new Task<Void>(){

            @Override
            protected Void call() throws Exception {
                PdfJoinerController.this.mergeService.mergeSections(PdfJoinerController.this.sections, outputFile, (current, total) -> Platform.runLater(() -> PdfJoinerController.this.joinButton.setText("Joining... " + current + "/" + total)));
                return null;
            }
        };
        joinTask.setOnSucceeded(event -> {
            logger.info("Join completed successfully");
            this.joinButton.setText("Join PDF");
            this.joinButton.setDisable(false);
            this.showInfo("PDF joined successfully: " + outputFile.getName());
        });
        joinTask.setOnFailed(event -> {
            logger.error("Join failed", joinTask.getException());
            this.joinButton.setText("Join PDF");
            this.joinButton.setDisable(false);
            this.showError("Join failed: " + joinTask.getException().getMessage());
        });
        new Thread(joinTask).start();
    }

    private void showError(String message) {
        logger.error("Showing error notification: {}", (Object)message);
        this.showNotification(message, "#d32f2f");
    }

    private void showWarning(String message) {
        logger.warn("Showing warning notification: {}", (Object)message);
        this.showNotification(message, "#f57c00");
    }

    private void showInfo(String message) {
        logger.info("Showing info notification: {}", (Object)message);
        this.showNotification(message, "#388e3c");
    }

    private void showNotification(String message, String bgColor) {
        Platform.runLater(() -> {
            this.notificationLabel.setText(message);
            this.notificationLabel.setStyle("-fx-padding: 8; -fx-font-size: 12; -fx-text-fill: white; -fx-background-color: " + bgColor + ";");
            this.notificationLabel.setVisible(true);
            Timeline timeline = new Timeline(new KeyFrame(Duration.seconds(5.0), event -> this.notificationLabel.setVisible(false), new KeyValue[0]));
            timeline.play();
        });
    }
}

