/*
 * Decompiled with CFR 0.152.
 */
package com.datmt.pdftools.ui.renamer;

import com.datmt.pdftools.model.RenameItem;
import com.datmt.pdftools.service.PdfTitleExtractor;
import com.datmt.pdftools.util.CreditLinkHandler;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Stream;
import javafx.application.Platform;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.value.ChangeListener;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.fxml.FXML;
import javafx.scene.control.Alert;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonType;
import javafx.scene.control.CheckBox;
import javafx.scene.control.Hyperlink;
import javafx.scene.control.Label;
import javafx.scene.control.ProgressBar;
import javafx.scene.control.SelectionMode;
import javafx.scene.control.TableCell;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.cell.CheckBoxTableCell;
import javafx.scene.control.cell.TextFieldTableCell;
import javafx.scene.layout.HBox;
import javafx.stage.DirectoryChooser;
import javafx.stage.FileChooser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfBulkRenamerController {
    private static final Logger logger = LoggerFactory.getLogger(PdfBulkRenamerController.class);
    @FXML
    private Button addFilesButton;
    @FXML
    private Button addFolderButton;
    @FXML
    private CheckBox includeSubfoldersCheck;
    @FXML
    private Button removeSelectedButton;
    @FXML
    private Button clearAllButton;
    @FXML
    private Label statusLabel;
    @FXML
    private TableView<RenameItem> fileTable;
    @FXML
    private TableColumn<RenameItem, Boolean> selectColumn;
    @FXML
    private TableColumn<RenameItem, String> originalNameColumn;
    @FXML
    private TableColumn<RenameItem, String> extractedTitleColumn;
    @FXML
    private TableColumn<RenameItem, String> newFilenameColumn;
    @FXML
    private TableColumn<RenameItem, RenameItem.RenameStatus> statusColumn;
    @FXML
    private CheckBox replaceSpacesCheck;
    @FXML
    private CheckBox normalizeAsciiCheck;
    @FXML
    private CheckBox lowercaseCheck;
    @FXML
    private HBox progressSection;
    @FXML
    private ProgressBar progressBar;
    @FXML
    private Label progressLabel;
    @FXML
    private Label statsLabel;
    @FXML
    private Button scanButton;
    @FXML
    private Button renameButton;
    @FXML
    private HBox resultsSection;
    @FXML
    private Label resultsLabel;
    @FXML
    private Hyperlink creditLink;
    private final ObservableList<RenameItem> items = FXCollections.observableArrayList();
    private final Map<RenameItem, SimpleBooleanProperty> selectionMap = new HashMap<RenameItem, SimpleBooleanProperty>();
    private final PdfTitleExtractor titleExtractor = new PdfTitleExtractor();
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    private boolean isProcessing = false;

    @FXML
    public void initialize() {
        logger.info("Initializing PDF Bulk Renamer");
        this.setupWindowCloseHandler();
        this.setupTable();
        CreditLinkHandler.setup(this.creditLink);
        this.updateUI();
    }

    private void setupWindowCloseHandler() {
        Platform.runLater(() -> {
            if (this.addFilesButton.getScene() != null && this.addFilesButton.getScene().getWindow() != null) {
                this.addFilesButton.getScene().getWindow().setOnCloseRequest(event -> this.cleanup());
            }
        });
    }

    private void setupTable() {
        this.fileTable.setItems(this.items);
        this.fileTable.setEditable(true);
        this.fileTable.getSelectionModel().setSelectionMode(SelectionMode.MULTIPLE);
        this.selectColumn.setCellFactory(col -> new CheckBoxTableCell());
        this.selectColumn.setCellValueFactory(cellData -> {
            RenameItem item = (RenameItem)cellData.getValue();
            SimpleBooleanProperty prop = this.selectionMap.computeIfAbsent(item, k -> new SimpleBooleanProperty(true));
            prop.addListener((ChangeListener<? super Boolean>)((ChangeListener<Boolean>)(obs, oldVal, newVal) -> this.updateUI()));
            return prop;
        });
        this.originalNameColumn.setCellValueFactory(cellData -> new SimpleStringProperty(((RenameItem)cellData.getValue()).getOriginalFilename()));
        this.extractedTitleColumn.setCellValueFactory(cellData -> new SimpleStringProperty(((RenameItem)cellData.getValue()).getExtractedTitle() != null ? ((RenameItem)cellData.getValue()).getExtractedTitle() : "(no title found)"));
        this.extractedTitleColumn.setCellFactory(col -> new TableCell<RenameItem, String>(this){

            @Override
            protected void updateItem(String item, boolean empty) {
                super.updateItem(item, empty);
                if (empty || item == null) {
                    this.setText(null);
                    this.setStyle("");
                } else {
                    this.setText(item);
                    if (item.equals("(no title found)")) {
                        this.setStyle("-fx-text-fill: #888888; -fx-font-style: italic;");
                    } else {
                        this.setStyle("");
                    }
                }
            }
        });
        this.newFilenameColumn.setCellValueFactory(cellData -> new SimpleStringProperty(((RenameItem)cellData.getValue()).getNewFilename()));
        this.newFilenameColumn.setCellFactory(TextFieldTableCell.forTableColumn());
        this.newFilenameColumn.setOnEditCommit(event -> {
            RenameItem item = (RenameItem)event.getRowValue();
            Object newValue = (String)event.getNewValue();
            if (newValue != null && !((String)newValue).isBlank()) {
                if (!((String)newValue).toLowerCase().endsWith(".pdf")) {
                    newValue = (String)newValue + ".pdf";
                }
                item.setNewFilename((String)newValue);
                if (item.getStatus() == RenameItem.RenameStatus.NO_TITLE) {
                    item.setStatus(RenameItem.RenameStatus.READY);
                }
            }
            this.fileTable.refresh();
            this.updateUI();
        });
        this.statusColumn.setCellValueFactory(cellData -> new SimpleObjectProperty<RenameItem.RenameStatus>(((RenameItem)cellData.getValue()).getStatus()));
        this.statusColumn.setCellFactory(col -> new TableCell<RenameItem, RenameItem.RenameStatus>(this){

            @Override
            protected void updateItem(RenameItem.RenameStatus status, boolean empty) {
                super.updateItem(status, empty);
                if (empty || status == null) {
                    this.setText(null);
                    this.setStyle("");
                } else {
                    switch (status) {
                        case PENDING: {
                            this.setText("Pending");
                            this.setStyle("-fx-text-fill: #888888;");
                            break;
                        }
                        case READY: {
                            this.setText("Ready");
                            this.setStyle("-fx-text-fill: #2196F3;");
                            break;
                        }
                        case NO_TITLE: {
                            this.setText("No Title");
                            this.setStyle("-fx-text-fill: #ff9800;");
                            break;
                        }
                        case SUCCESS: {
                            this.setText("Renamed");
                            this.setStyle("-fx-text-fill: #4CAF50; -fx-font-weight: bold;");
                            break;
                        }
                        case SKIPPED: {
                            this.setText("Skipped");
                            this.setStyle("-fx-text-fill: #9e9e9e;");
                            break;
                        }
                        case ERROR: {
                            this.setText("Error");
                            this.setStyle("-fx-text-fill: #f44336;");
                        }
                    }
                }
            }
        });
    }

    @FXML
    private void onAddFiles() {
        FileChooser fileChooser = new FileChooser();
        fileChooser.setTitle("Select PDF Files");
        fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PDF Files", "*.pdf"));
        List<File> files = fileChooser.showOpenMultipleDialog(this.addFilesButton.getScene().getWindow());
        if (files != null) {
            for (File file : files) {
                this.addFile(file);
            }
            this.updateUI();
        }
    }

    @FXML
    private void onAddFolder() {
        DirectoryChooser dirChooser = new DirectoryChooser();
        dirChooser.setTitle("Select Folder with PDF Files");
        File directory = dirChooser.showDialog(this.addFolderButton.getScene().getWindow());
        if (directory != null) {
            this.addFilesFromDirectory(directory, this.includeSubfoldersCheck.isSelected());
            this.updateUI();
        }
    }

    private void addFilesFromDirectory(File directory, boolean includeSubfolders) {
        logger.info("Scanning directory: {} (subfolders: {})", (Object)directory.getAbsolutePath(), (Object)includeSubfolders);
        try {
            int maxDepth = includeSubfolders ? Integer.MAX_VALUE : 1;
            try (Stream<Path> paths = Files.walk(directory.toPath(), maxDepth, new FileVisitOption[0]);){
                paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(p -> p.toString().toLowerCase().endsWith(".pdf")).map(Path::toFile).forEach(this::addFile);
            }
        }
        catch (IOException e) {
            logger.error("Failed to scan directory: {}", (Object)e.getMessage());
            this.showError("Failed to scan directory: " + e.getMessage());
        }
    }

    private void addFile(File file) {
        for (RenameItem item : this.items) {
            if (!item.getOriginalFile().getAbsolutePath().equals(file.getAbsolutePath())) continue;
            logger.debug("File already added: {}", (Object)file.getName());
            return;
        }
        RenameItem item = new RenameItem(file);
        this.items.add(item);
        this.selectionMap.put(item, new SimpleBooleanProperty(true));
        logger.debug("Added file: {}", (Object)file.getName());
    }

    @FXML
    private void onRemoveSelected() {
        ArrayList<RenameItem> toRemove = new ArrayList<RenameItem>();
        for (RenameItem item : this.items) {
            SimpleBooleanProperty selected = this.selectionMap.get(item);
            if (selected == null || !selected.get()) continue;
            toRemove.add(item);
        }
        this.items.removeAll((Collection<?>)toRemove);
        for (RenameItem item : toRemove) {
            this.selectionMap.remove(item);
        }
        this.updateUI();
    }

    @FXML
    private void onClearAll() {
        this.items.clear();
        this.selectionMap.clear();
        this.resultsSection.setVisible(false);
        this.resultsSection.setManaged(false);
        this.updateUI();
    }

    private PdfTitleExtractor.NormalizationOptions buildNormalizationOptions() {
        return new PdfTitleExtractor.NormalizationOptions().setReplaceSpacesWithHyphens(this.replaceSpacesCheck.isSelected()).setNormalizeToAscii(this.normalizeAsciiCheck.isSelected()).setLowercase(this.lowercaseCheck.isSelected());
    }

    @FXML
    private void onScanTitles() {
        if (this.items.isEmpty() || this.isProcessing) {
            return;
        }
        this.isProcessing = true;
        this.updateUI();
        this.progressSection.setVisible(true);
        this.progressSection.setManaged(true);
        this.progressBar.setProgress(0.0);
        ArrayList<RenameItem> itemsToScan = new ArrayList<RenameItem>(this.items);
        int total = itemsToScan.size();
        PdfTitleExtractor.NormalizationOptions options = this.buildNormalizationOptions();
        this.executor.submit(() -> {
            for (int i = 0; i < total; ++i) {
                RenameItem item = (RenameItem)itemsToScan.get(i);
                int index = i;
                Platform.runLater(() -> {
                    this.progressLabel.setText("Scanning: " + item.getOriginalFilename() + " (" + (index + 1) + "/" + total + ")");
                    this.progressBar.setProgress((double)index / (double)total);
                });
                String title = this.titleExtractor.extractTitle(item.getOriginalFile());
                if (title != null && !title.isBlank()) {
                    String filename = this.titleExtractor.generateFilename(title, options);
                    String uniqueFilename = this.titleExtractor.generateUniqueFilename(item.getOriginalFile().getParentFile(), filename);
                    item.setExtractedTitle(title);
                    if (uniqueFilename != null) {
                        item.setNewFilename(uniqueFilename);
                        item.setStatus(RenameItem.RenameStatus.READY);
                        continue;
                    }
                    item.setNewFilename(item.getOriginalFilename());
                    item.setStatus(RenameItem.RenameStatus.NO_TITLE);
                    item.setErrorMessage("Could not generate unique filename");
                    continue;
                }
                item.setExtractedTitle(null);
                item.setNewFilename(item.getOriginalFilename());
                item.setStatus(RenameItem.RenameStatus.NO_TITLE);
            }
            Platform.runLater(() -> {
                this.progressSection.setVisible(false);
                this.progressSection.setManaged(false);
                this.isProcessing = false;
                this.fileTable.refresh();
                this.updateUI();
                logger.info("Title scanning complete: {} files processed", (Object)total);
            });
        });
    }

    @FXML
    private void onRenameAll() {
        if (this.items.isEmpty() || this.isProcessing) {
            return;
        }
        ArrayList<RenameItem> itemsToRename = new ArrayList<RenameItem>();
        for (RenameItem item : this.items) {
            SimpleBooleanProperty selected = this.selectionMap.get(item);
            if (selected == null || !selected.get() || item.getStatus() != RenameItem.RenameStatus.READY || !item.needsRename()) continue;
            itemsToRename.add(item);
        }
        if (itemsToRename.isEmpty()) {
            this.showError("No files are ready to rename. Click 'Scan Titles' first.");
            return;
        }
        Alert confirm = new Alert(Alert.AlertType.CONFIRMATION);
        confirm.setTitle("Confirm Rename");
        confirm.setHeaderText("Rename " + itemsToRename.size() + " files?");
        confirm.setContentText("This will rename the selected PDF files based on their extracted titles.");
        Optional result = confirm.showAndWait();
        if (result.isEmpty() || result.get() != ButtonType.OK) {
            return;
        }
        this.isProcessing = true;
        this.updateUI();
        this.progressSection.setVisible(true);
        this.progressSection.setManaged(true);
        this.progressBar.setProgress(0.0);
        int total = itemsToRename.size();
        this.executor.submit(() -> {
            int successCount = 0;
            int errorCount = 0;
            for (int i = 0; i < total; ++i) {
                RenameItem item = (RenameItem)itemsToRename.get(i);
                int index = i;
                Platform.runLater(() -> {
                    this.progressLabel.setText("Renaming: " + item.getOriginalFilename() + " (" + (index + 1) + "/" + total + ")");
                    this.progressBar.setProgress((double)index / (double)total);
                });
                try {
                    String unique;
                    File targetFile = item.getTargetFile();
                    if (targetFile.exists() && (unique = this.titleExtractor.generateUniqueFilename(item.getOriginalFile().getParentFile(), item.getNewFilename())) != null) {
                        item.setNewFilename(unique);
                        targetFile = item.getTargetFile();
                    }
                    if (item.getOriginalFile().renameTo(targetFile)) {
                        item.setStatus(RenameItem.RenameStatus.SUCCESS);
                        ++successCount;
                        logger.info("Renamed: {} -> {}", (Object)item.getOriginalFilename(), (Object)item.getNewFilename());
                        continue;
                    }
                    item.setStatus(RenameItem.RenameStatus.ERROR);
                    item.setErrorMessage("Rename failed");
                    ++errorCount;
                    logger.warn("Failed to rename: {}", (Object)item.getOriginalFilename());
                    continue;
                }
                catch (Exception e) {
                    item.setStatus(RenameItem.RenameStatus.ERROR);
                    item.setErrorMessage(e.getMessage());
                    ++errorCount;
                    logger.error("Error renaming {}: {}", (Object)item.getOriginalFilename(), (Object)e.getMessage());
                }
            }
            int finalSuccess = successCount;
            int finalError = errorCount;
            Platform.runLater(() -> {
                this.progressSection.setVisible(false);
                this.progressSection.setManaged(false);
                this.isProcessing = false;
                this.fileTable.refresh();
                this.updateUI();
                this.resultsSection.setVisible(true);
                this.resultsSection.setManaged(true);
                if (finalError == 0) {
                    this.resultsLabel.setText("Successfully renamed " + finalSuccess + " files!");
                    this.resultsSection.setStyle("-fx-padding: 15; -fx-background-color: #e8f5e9; -fx-border-color: #4CAF50; -fx-border-radius: 4;");
                    this.resultsLabel.setStyle("-fx-font-size: 14; -fx-font-weight: bold; -fx-text-fill: #2e7d32;");
                } else {
                    this.resultsLabel.setText("Renamed " + finalSuccess + " files, " + finalError + " errors");
                    this.resultsSection.setStyle("-fx-padding: 15; -fx-background-color: #fff3e0; -fx-border-color: #ff9800; -fx-border-radius: 4;");
                    this.resultsLabel.setStyle("-fx-font-size: 14; -fx-font-weight: bold; -fx-text-fill: #e65100;");
                }
                logger.info("Rename complete: {} success, {} errors", (Object)finalSuccess, (Object)finalError);
            });
        });
    }

    private void updateUI() {
        boolean hasFiles = !this.items.isEmpty();
        int selectedCount = 0;
        int readyCount = 0;
        for (RenameItem item : this.items) {
            SimpleBooleanProperty selected = this.selectionMap.get(item);
            if (selected == null || !selected.get()) continue;
            ++selectedCount;
            if (item.getStatus() != RenameItem.RenameStatus.READY || !item.needsRename()) continue;
            ++readyCount;
        }
        this.addFilesButton.setDisable(this.isProcessing);
        this.addFolderButton.setDisable(this.isProcessing);
        this.removeSelectedButton.setDisable(!hasFiles || this.isProcessing);
        this.clearAllButton.setDisable(!hasFiles || this.isProcessing);
        this.scanButton.setDisable(!hasFiles || this.isProcessing);
        this.renameButton.setDisable(readyCount == 0 || this.isProcessing);
        if (this.isProcessing) {
            this.statusLabel.setText("Processing...");
        } else if (hasFiles) {
            this.statusLabel.setText(this.items.size() + " files loaded");
        } else {
            this.statusLabel.setText("No files loaded");
        }
        this.statsLabel.setText(this.items.size() + " files | " + readyCount + " ready to rename");
    }

    private void showError(String message) {
        Alert alert = new Alert(Alert.AlertType.ERROR);
        alert.setTitle("Error");
        alert.setHeaderText(null);
        alert.setContentText(message);
        alert.showAndWait();
    }

    public void cleanup() {
        logger.info("Cleaning up PDF Bulk Renamer");
        this.executor.shutdownNow();
        this.items.clear();
        this.selectionMap.clear();
    }
}

